<?php
/**
 * Admin functionality
 */

if (!defined('ABSPATH')) {
    exit;
}

class WPBD_Admin
{

    public function __construct()
    {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'settings_init'));
        add_action('add_meta_boxes', array($this, 'add_meta_boxes'));
        add_action('save_post', array($this, 'save_business_meta'));
        add_action('show_user_profile', array($this, 'extra_user_profile_fields'));
        add_action('edit_user_profile', array($this, 'extra_user_profile_fields'));
        add_action('personal_options_update', array($this, 'save_extra_user_profile_fields'));
        add_action('edit_user_profile_update', array($this, 'save_extra_user_profile_fields'));
    }

    /**
     * Add admin menu pages
     */
    public function add_admin_menu()
    {
        add_submenu_page(
            'edit.php?post_type=wpbd_business',
            __('Settings', 'wp-business-directory'),
            __('Settings', 'wp-business-directory'),
            'manage_options',
            'wpbd-settings',
            array($this, 'settings_page')
        );

        add_submenu_page(
            'edit.php?post_type=wpbd_business',
            __('User Approvals', 'wp-business-directory'),
            __('User Approvals', 'wp-business-directory'),
            'manage_options',
            'wpbd-user-approvals',
            array($this, 'user_approvals_page')
        );
    }

    /**
     * Initialize settings
     */
    public function settings_init()
    {
        register_setting('wpbd_settings', 'wpbd_user_approval_required');
        register_setting('wpbd_settings', 'wpbd_default_listings_limit');
        register_setting('wpbd_settings', 'wpbd_custom_css');

        add_settings_section(
            'wpbd_settings_section',
            __('Business Directory Settings', 'wp-business-directory'),
            array($this, 'settings_section_callback'),
            'wpbd_settings'
        );

        add_settings_field(
            'wpbd_user_approval_required',
            __('Require Admin Approval', 'wp-business-directory'),
            array($this, 'user_approval_render'),
            'wpbd_settings',
            'wpbd_settings_section'
        );

        add_settings_field(
            'wpbd_default_listings_limit',
            __('Default Listings Limit per User', 'wp-business-directory'),
            array($this, 'listings_limit_render'),
            'wpbd_settings',
            'wpbd_settings_section'
        );

        add_settings_field(
            'wpbd_custom_css',
            __('Custom CSS', 'wp-business-directory'),
            array($this, 'custom_css_render'),
            'wpbd_settings',
            'wpbd_settings_section'
        );
    }

    /**
     * Settings section callback
     */
    public function settings_section_callback()
    {
        echo __('Manage your business directory settings.', 'wp-business-directory');
    }

    /**
     * User approval setting field
     */
    public function user_approval_render()
    {
        $option = get_option('wpbd_user_approval_required');
        ?>
        <select name="wpbd_user_approval_required">
            <option value="yes" <?php selected($option, 'yes'); ?>><?php _e('Yes', 'wp-business-directory'); ?></option>
            <option value="no" <?php selected($option, 'no'); ?>><?php _e('No', 'wp-business-directory'); ?></option>
        </select>
        <?php
    }

    /**
     * Listings limit setting field
     */
    public function listings_limit_render()
    {
        $option = get_option('wpbd_default_listings_limit', 5);
        ?>
        <input type="number" name="wpbd_default_listings_limit" value="<?php echo esc_attr($option); ?>" min="1" />
        <?php
    }

    /**
     * Custom CSS setting field
     */
    public function custom_css_render()
    {
        $custom_css = get_option('wpbd_custom_css', '');
        $template_url = WPBD_PLUGIN_URL . 'assets/css/default-styles-template.css';
        ?>
        <div class="wpbd-custom-css-editor">
            <p class="description">
                <?php _e('Add custom CSS to override default plugin styles. Changes will be applied to the frontend.', 'wp-business-directory'); ?>
            </p>
            <p>
                <button type="button" class="button" id="wpbd-load-template">
                    <?php _e('Load Default Styles Template', 'wp-business-directory'); ?>
                </button>
                <button type="button" class="button" id="wpbd-reset-css">
                    <?php _e('Clear Custom CSS', 'wp-business-directory'); ?>
                </button>
            </p>
            <textarea name="wpbd_custom_css" id="wpbd-custom-css" rows="20"
                style="width: 100%; font-family: monospace; font-size: 13px;"><?php echo esc_textarea($custom_css); ?></textarea>
        </div>
        <script>
            jQuery(document).ready(function ($) {
                $('#wpbd-load-template').on('click', function () {
                    if (confirm('<?php _e('This will replace the current content with the default styles template. Continue?', 'wp-business-directory'); ?>')) {
                        $.get('<?php echo esc_js($template_url); ?>', function (data) {
                            $('#wpbd-custom-css').val(data);
                        });
                    }
                });

                $('#wpbd-reset-css').on('click', function () {
                    if (confirm('<?php _e('This will clear all custom CSS. Continue?', 'wp-business-directory'); ?>')) {
                        $('#wpbd-custom-css').val('');
                    }
                });
            });
        </script>
        <style>
            .wpbd-custom-css-editor {
                max-width: 800px;
            }

            .wpbd-custom-css-editor textarea {
                border: 1px solid #ddd;
                border-radius: 4px;
                padding: 10px;
            }
        </style>
        <?php
    }
    /**
     * Settings page content
     */
    public function settings_page()
    {
        ?>
        <div class="wrap">
            <h1><?php _e('Business Directory Settings', 'wp-business-directory'); ?></h1>
            <form method="post" action="options.php">
                <?php
                settings_fields('wpbd_settings');
                do_settings_sections('wpbd_settings');
                submit_button();
                ?>
            </form>
        </div>
        <?php
    }

    /**
     * Add meta boxes for business listings
     */
    public function add_meta_boxes()
    {
        add_meta_box(
            'wpbd-business-details',
            __('Business Details', 'wp-business-directory'),
            array($this, 'business_details_meta_box'),
            'wpbd_business',
            'normal',
            'high'
        );
    }

    /**
     * Business details meta box
     */
    public function business_details_meta_box($post)
    {
        wp_nonce_field('wpbd_save_business_meta', 'wpbd_business_meta_nonce');

        $phone = get_post_meta($post->ID, '_wpbd_phone', true);
        $address = get_post_meta($post->ID, '_wpbd_address', true);
        $website = get_post_meta($post->ID, '_wpbd_website', true);
        $description = get_post_meta($post->ID, '_wpbd_description', true);
        $naics_code = get_post_meta($post->ID, '_wpbd_naics_code', true);
        $featured = get_post_meta($post->ID, '_wpbd_featured', true);
        $user_id = get_post_meta($post->ID, '_wpbd_user_id', true);

        ?>
        <table class="form-table">
            <tr>
                <th><label for="wpbd_description"><?php _e('Description', 'wp-business-directory'); ?></label></th>
                <td>
                    <textarea id="wpbd_description" name="wpbd_description" rows="3" cols="50"
                        maxlength="150"><?php echo esc_textarea($description); ?></textarea>
                    <p class="description"><?php _e('Limited to 150 characters.', 'wp-business-directory'); ?></p>
                </td>
            </tr>
            <tr>
                <th><label for="wpbd_phone"><?php _e('Phone Number', 'wp-business-directory'); ?></label></th>
                <td><input type="text" id="wpbd_phone" name="wpbd_phone" value="<?php echo esc_attr($phone); ?>" size="30" />
                </td>
            </tr>
            <tr>
                <th><label for="wpbd_address"><?php _e('Address', 'wp-business-directory'); ?></label></th>
                <td><input type="text" id="wpbd_address" name="wpbd_address" value="<?php echo esc_attr($address); ?>"
                        size="30" /></td>
            </tr>
            <tr>
                <th><label for="wpbd_website"><?php _e('Website', 'wp-business-directory'); ?></label></th>
                <td><input type="url" id="wpbd_website" name="wpbd_website" value="<?php echo esc_url($website); ?>"
                        size="30" /></td>
            </tr>
            <tr>
                <th><label for="wpbd_naics_code"><?php _e('NAICS Code', 'wp-business-directory'); ?></label></th>
                <td><input type="text" id="wpbd_naics_code" name="wpbd_naics_code" value="<?php echo esc_attr($naics_code); ?>"
                        size="30" /></td>
            </tr>
            <?php if (current_user_can('manage_options')): ?>
                <tr>
                    <th><label for="wpbd_featured"><?php _e('Featured Listing', 'wp-business-directory'); ?></label></th>
                    <td>
                        <input type="checkbox" id="wpbd_featured" name="wpbd_featured" value="1" <?php checked($featured, 1); ?> />
                        <span
                            class="description"><?php _e('Feature this listing on the main directory page.', 'wp-business-directory'); ?></span>
                    </td>
                </tr>
            <?php endif; ?>
            <?php if ($user_id): ?>
                <tr>
                    <th><label><?php _e('Submitted by User', 'wp-business-directory'); ?></label></th>
                    <td><?php echo get_userdata($user_id)->display_name; ?> (ID: <?php echo $user_id; ?>)</td>
                </tr>
            <?php endif; ?>
        </table>
        <?php
    }

    /**
     * Save business meta data
     */
    public function save_business_meta($post_id)
    {
        // Verify nonce
        if (!isset($_POST['wpbd_business_meta_nonce']) || !wp_verify_nonce($_POST['wpbd_business_meta_nonce'], 'wpbd_save_business_meta')) {
            return;
        }

        // Check if user has permission to edit
        if (!current_user_can('edit_post', $post_id)) {
            return;
        }

        // Save meta fields
        if (isset($_POST['wpbd_description'])) {
            update_post_meta($post_id, '_wpbd_description', sanitize_text_field($_POST['wpbd_description']));
        }

        if (isset($_POST['wpbd_phone'])) {
            update_post_meta($post_id, '_wpbd_phone', sanitize_text_field($_POST['wpbd_phone']));
        }

        if (isset($_POST['wpbd_address'])) {
            update_post_meta($post_id, '_wpbd_address', sanitize_text_field($_POST['wpbd_address']));
        }

        if (isset($_POST['wpbd_website'])) {
            update_post_meta($post_id, '_wpbd_website', esc_url_raw($_POST['wpbd_website']));
        }

        if (isset($_POST['wpbd_naics_code'])) {
            update_post_meta($post_id, '_wpbd_naics_code', sanitize_text_field($_POST['wpbd_naics_code']));
        }

        if (current_user_can('manage_options')) {
            update_post_meta($post_id, '_wpbd_featured', isset($_POST['wpbd_featured']) ? 1 : 0);
        }
    }

    /**
     * Add extra user profile fields
     */
    public function extra_user_profile_fields($user)
    {
        $approved = get_user_meta($user->ID, 'wpbd_approved', true);
        $listings_limit = get_user_meta($user->ID, 'wpbd_listings_limit', true);
        ?>
        <h3><?php _e('Business Directory Settings', 'wp-business-directory'); ?></h3>
        <table class="form-table">
            <tr>
                <th><label for="wpbd_approved"><?php _e('Approved for Business Listings', 'wp-business-directory'); ?></label>
                </th>
                <td>
                    <input type="checkbox" name="wpbd_approved" id="wpbd_approved" value="1" <?php checked($approved, 1); ?> />
                    <span
                        class="description"><?php _e('Check to approve this user for submitting business listings.', 'wp-business-directory'); ?></span>
                </td>
            </tr>
            <tr>
                <th><label for="wpbd_listings_limit"><?php _e('Business Listings Limit', 'wp-business-directory'); ?></label>
                </th>
                <td>
                    <input type="number" name="wpbd_listings_limit" id="wpbd_listings_limit"
                        value="<?php echo esc_attr($listings_limit); ?>" class="regular-text" />
                    <span
                        class="description"><?php _e('Maximum number of business listings this user can submit.', 'wp-business-directory'); ?></span>
                </td>
            </tr>
        </table>
        <?php
    }

    /**
     * Save extra user profile fields
     */
    public function save_extra_user_profile_fields($user_id)
    {
        if (!current_user_can('edit_user', $user_id)) {
            return false;
        }

        update_user_meta($user_id, 'wpbd_approved', isset($_POST['wpbd_approved']) ? 1 : 0);
        update_user_meta($user_id, 'wpbd_listings_limit', sanitize_text_field($_POST['wpbd_listings_limit']));
    }

    /**
     * User approvals page content
     */
    public function user_approvals_page()
    {
        // Handle approval actions
        if (isset($_POST['approve_user']) && wp_verify_nonce($_POST['wpbd_user_approval_nonce'], 'wpbd_user_approval')) {
            $user_id = intval($_POST['user_id']);
            WPBD_Install::approve_user($user_id);
            echo '<div class="notice notice-success"><p>User approved successfully.</p></div>';
        }

        if (isset($_POST['reject_user']) && wp_verify_nonce($_POST['wpbd_user_approval_nonce'], 'wpbd_user_approval')) {
            $user_id = intval($_POST['user_id']);
            update_user_meta($user_id, 'wpbd_approved', 0);
            echo '<div class="notice notice-success"><p>User rejected successfully.</p></div>';
        }

        // Get users who are not approved but have submitted businesses or registered
        $users = get_users(array(
            'meta_query' => array(
                array(
                    'key' => 'wpbd_approved',
                    'value' => '0',
                    'compare' => '='
                )
            )
        ));

        ?>
        <div class="wrap">
            <h1><?php _e('User Approvals', 'wp-business-directory'); ?></h1>

            <?php if (empty($users)): ?>
                <div class="notice notice-info">
                    <p><?php _e('No users pending approval.', 'wp-business-directory'); ?></p>
                </div>
            <?php else: ?>
                <table class="wp-list-table widefat fixed striped">
                    <thead>
                        <tr>
                            <th><?php _e('Username', 'wp-business-directory'); ?></th>
                            <th><?php _e('Email', 'wp-business-directory'); ?></th>
                            <th><?php _e('Registered', 'wp-business-directory'); ?></th>
                            <th><?php _e('Business Listings', 'wp-business-directory'); ?></th>
                            <th><?php _e('Actions', 'wp-business-directory'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($users as $user):
                            $listings_count = count_user_posts($user->ID, 'wpbd_business');
                            ?>
                            <tr>
                                <td><?php echo esc_html($user->user_login); ?></td>
                                <td><?php echo esc_html($user->user_email); ?></td>
                                <td><?php echo esc_html($user->user_registered); ?></td>
                                <td><?php echo $listings_count; ?></td>
                                <td>
                                    <form method="post" style="display: inline-block;">
                                        <?php wp_nonce_field('wpbd_user_approval', 'wpbd_user_approval_nonce'); ?>
                                        <input type="hidden" name="user_id" value="<?php echo $user->ID; ?>">
                                        <input type="submit" name="approve_user" class="button button-primary"
                                            value="<?php _e('Approve', 'wp-business-directory'); ?>">
                                    </form>
                                    <form method="post" style="display: inline-block;">
                                        <?php wp_nonce_field('wpbd_user_approval', 'wpbd_user_approval_nonce'); ?>
                                        <input type="hidden" name="user_id" value="<?php echo $user->ID; ?>">
                                        <input type="submit" name="reject_user" class="button"
                                            value="<?php _e('Reject', 'wp-business-directory'); ?>">
                                    </form>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php endif; ?>
        </div>
        <?php
    }
}
