<?php
/**
 * Public functionality
 */

if (!defined('ABSPATH')) {
    exit;
}

class WPBD_Public
{

    public function __construct()
    {
        add_action('wp_ajax_wpbd_load_category_listings', array($this, 'load_category_listings'));
        add_action('wp_ajax_nopriv_wpbd_load_category_listings', array($this, 'load_category_listings'));
        add_action('wp_ajax_wpbd_submit_business', array($this, 'submit_business'));
        add_action('wp_ajax_wpbd_delete_business', array($this, 'delete_business'));
        add_action('wp_ajax_wpbd_get_user_businesses', array($this, 'get_user_businesses'));
        add_action('wp_ajax_wpbd_get_business_data', array($this, 'get_business_data'));
        add_action('wp_ajax_nopriv_wpbd_login', array($this, 'handle_login'));
        add_action('wp_ajax_nopriv_wpbd_register', array($this, 'handle_register'));
        add_action('wp_ajax_wpbd_search_listings', array($this, 'search_listings'));
        add_action('wp_ajax_nopriv_wpbd_search_listings', array($this, 'search_listings'));
        add_action('wp_ajax_wpbd_get_business_html', array($this, 'get_business_html'));
        add_action('wp_ajax_nopriv_wpbd_get_business_html', array($this, 'get_business_html'));
    }

    /**
     * Handle AJAX request to load category listings
     */
    public function load_category_listings()
    {
        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'], 'wpbd_nonce')) {
            wp_die('Security check failed');
        }

        $category_id = intval($_POST['category_id']);
        $page = intval($_POST['page']);

        // Get listings in this category
        $args = array(
            'post_type' => 'wpbd_business',
            'post_status' => 'publish',
            'posts_per_page' => 10,
            'paged' => $page,
            'tax_query' => array(
                array(
                    'taxonomy' => 'wpbd_category',
                    'field' => 'term_id',
                    'terms' => $category_id,
                ),
            ),
        );

        $query = new WP_Query($args);

        $html = '';
        if ($query->have_posts()) {
            while ($query->have_posts()) {
                $query->the_post();
                $html .= $this->get_business_listing_html(get_the_ID());
            }
        } else {
            $html = '<p>' . __('No businesses found in this category.', 'wp-business-directory') . '</p>';
        }

        wp_reset_postdata();

        wp_send_json_success(array(
            'html' => $html,
            'max_num_pages' => $query->max_num_pages,
            'current_page' => $page
        ));
    }

    /**
     * Generate HTML for a business listing
     */
    private function get_business_listing_html($post_id)
    {
        $description = get_post_meta($post_id, '_wpbd_description', true);
        $phone = get_post_meta($post_id, '_wpbd_phone', true);
        $address = get_post_meta($post_id, '_wpbd_address', true);
        $website = get_post_meta($post_id, '_wpbd_website', true);

        $html = '<div class="wpbd-business-listing">';

        // Logo/Featured image above title
        if (has_post_thumbnail($post_id)) {
            $html .= '<div class="wpbd-business-logo">' . get_the_post_thumbnail($post_id, 'medium') . '</div>';
        }

        // Business title
        $html .= '<h3 class="wpbd-business-title">' . get_the_title($post_id) . '</h3>';

        $html .= '<div class="wpbd-business-content">';

        if ($description) {
            $html .= '<p class="wpbd-description">' . esc_html($description) . '</p>';
        }

        if ($phone) {
            $html .= '<p class="wpbd-phone"><span class="dashicons dashicons-phone"></span> <a href="tel:' . esc_attr($phone) . '">' . esc_html($phone) . '</a></p>';
        }

        if ($address) {
            $html .= '<p class="wpbd-address"><span class="dashicons dashicons-location"></span> ' . esc_html($address) . '</p>';
        }

        if (isset($website) && $website) {
            $html .= '<p class="wpbd-website"><span class="dashicons dashicons-admin-site"></span> <a href="' . esc_url($website) . '" target="_blank" rel="noopener">' . __('Visit Website', 'wp-business-directory') . '</a></p>';
        }

        $html .= '</div>';
        $html .= '</div>';

        return $html;
    }


    /**
     * Handle business submission from frontend dashboard
     */
    public function submit_business()
    {
        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'], 'wpbd_nonce')) {
            wp_die('Security check failed');
        }

        // Check if user is logged in
        if (!is_user_logged_in()) {
            wp_send_json_error(__('You must be logged in to submit a business listing.', 'wp-business-directory'));
        }

        $user_id = get_current_user_id();

        // Check if user is approved or is an administrator
        $approved = get_user_meta($user_id, 'wpbd_approved', true);
        $is_admin = current_user_can('manage_options');

        if (!$approved && !$is_admin) {
            wp_send_json_error(__('Your account has not been approved yet.', 'wp-business-directory'));
        }

        // Check listing limit for new listings only
        $listings_limit = get_user_meta($user_id, 'wpbd_listings_limit', true);
        if (empty($listings_limit)) {
            $listings_limit = get_option('wpbd_default_listings_limit', 5);
        }

        // Get the post ID if editing
        $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;

        // For new listings, check the limit
        if (!$post_id) {
            $user_listings_count = count_user_posts($user_id, 'wpbd_business');
            if ($user_listings_count >= $listings_limit && !$is_admin) {
                wp_send_json_error(sprintf(__('You have reached your limit of %d listings.', 'wp-business-directory'), $listings_limit));
            }
        }

        // Process form data
        $title = sanitize_text_field($_POST['business_name']);
        $description = substr(sanitize_textarea_field($_POST['description']), 0, 150);
        $phone = sanitize_text_field($_POST['phone']);
        $address = sanitize_text_field($_POST['address']);
        $website = esc_url_raw($_POST['website']);
        $categories = array_map('intval', $_POST['categories']);

        // Set post status - publish for admins and approved users, pending for new submissions from regular users
        $post_status = $is_admin ? 'publish' : 'publish'; // All listings are published for approved users

        // If editing existing post
        if ($post_id) {
            // Check if user owns this listing
            $post = get_post($post_id);
            if ($post->post_author != $user_id && !current_user_can('edit_others_posts')) {
                wp_send_json_error(__('You do not have permission to edit this listing.', 'wp-business-directory'));
            }

            // Update post
            $updated_post_id = wp_update_post(array(
                'ID' => $post_id,
                'post_title' => $title,
                'post_content' => $description,
                'post_status' => $post_status,
            ));

            if (is_wp_error($updated_post_id)) {
                wp_send_json_error(__('Error updating listing. Please try again.', 'wp-business-directory'));
            }

            $post_id = $updated_post_id;
        } else {
            // Insert new post
            $post_id = wp_insert_post(array(
                'post_title' => $title,
                'post_content' => $description,
                'post_status' => $post_status,
                'post_type' => 'wpbd_business',
            ));

            if (is_wp_error($post_id)) {
                wp_send_json_error(__('Error submitting listing. Please try again.', 'wp-business-directory'));
            }

            // Save user ID to post meta
            update_post_meta($post_id, '_wpbd_user_id', $user_id);
        }

        // Save meta data
        update_post_meta($post_id, '_wpbd_description', $description);
        update_post_meta($post_id, '_wpbd_phone', $phone);
        update_post_meta($post_id, '_wpbd_address', $address);
        update_post_meta($post_id, '_wpbd_website', $website);

        // Set categories
        wp_set_post_terms($post_id, $categories, 'wpbd_category');

        // Handle logo upload if present
        if (!empty($_FILES['logo'])) {
            require_once(ABSPATH . 'wp-admin/includes/media.php');
            require_once(ABSPATH . 'wp-admin/includes/file.php');
            require_once(ABSPATH . 'wp-admin/includes/image.php');

            $attachment_id = media_handle_upload('logo', $post_id);

            if (!is_wp_error($attachment_id)) {
                set_post_thumbnail($post_id, $attachment_id);
            }
        }

        $message = $post_id ?
            __('Business listing updated successfully.', 'wp-business-directory') :
            __('Business listing submitted successfully.', 'wp-business-directory');

        wp_send_json_success($message);
    }

    /**
     * Handle business deletion from frontend dashboard
     */
    public function delete_business()
    {
        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'], 'wpbd_nonce')) {
            wp_die('Security check failed');
        }

        // Check if user is logged in
        if (!is_user_logged_in()) {
            wp_send_json_error(__('You must be logged in to delete a business listing.', 'wp-business-directory'));
        }

        $user_id = get_current_user_id();
        $post_id = intval($_POST['post_id']);

        // Check if user owns this listing
        $post = get_post($post_id);
        if ($post->post_author != $user_id && !current_user_can('delete_others_posts')) {
            wp_send_json_error(__('You do not have permission to delete this listing.', 'wp-business-directory'));
        }

        // Delete the post
        $result = wp_delete_post($post_id, true);

        if ($result) {
            wp_send_json_success(__('Business listing deleted successfully.', 'wp-business-directory'));
        } else {
            wp_send_json_error(__('Error deleting listing. Please try again.', 'wp-business-directory'));
        }
    }

    /**
     * Get user's businesses for dashboard
     */
    public function get_user_businesses()
    {
        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'], 'wpbd_nonce')) {
            wp_die('Security check failed');
        }

        // Check if user is logged in
        if (!is_user_logged_in()) {
            wp_send_json_error(__('You must be logged in to view your businesses.', 'wp-business-directory'));
        }

        $user_id = get_current_user_id();

        // Get user's businesses
        $args = array(
            'post_type' => 'wpbd_business',
            'post_status' => array('publish', 'pending', 'draft'),
            'author' => $user_id,
            'posts_per_page' => -1,
        );

        $query = new WP_Query($args);

        $businesses = array();
        if ($query->have_posts()) {
            while ($query->have_posts()) {
                $query->the_post();
                $post_id = get_the_ID();
                $businesses[] = array(
                    'ID' => $post_id,
                    'title' => get_the_title(),
                    'status' => get_post_status(),
                    'edit_link' => get_edit_post_link($post_id),
                );
            }
        }

        wp_reset_postdata();

        wp_send_json_success($businesses);
    }

    /**
     * Handle user login via AJAX
     */
    public function handle_login()
    {
        // Check nonce
        if (!wp_verify_nonce($_POST['nonce'], 'wpbd_nonce')) {
            wp_send_json_error(__('Security check failed.', 'wp-business-directory'));
        }

        // Get login credentials
        $username = sanitize_user($_POST['username']);
        $password = $_POST['password'];

        // Authenticate user
        $user = wp_signon(array(
            'user_login' => $username,
            'user_password' => $password,
            'remember' => isset($_POST['remember'])
        ));

        if (is_wp_error($user)) {
            wp_send_json_error($user->get_error_message());
        }

        wp_send_json_success(__('Login successful.', 'wp-business-directory'));
    }

    /**
     * Handle user registration via AJAX
     */
    public function handle_register()
    {
        // Check nonce
        if (!wp_verify_nonce($_POST['nonce'], 'wpbd_nonce')) {
            wp_send_json_error(__('Security check failed.', 'wp-business-directory'));
        }

        // Get registration data
        $username = sanitize_user($_POST['username']);
        $email = sanitize_email($_POST['email']);
        $password = $_POST['password'];

        // Validate input
        if (empty($username) || empty($email) || empty($password)) {
            wp_send_json_error(__('All fields are required.', 'wp-business-directory'));
        }

        // Check if user already exists
        if (username_exists($username)) {
            wp_send_json_error(__('Username already exists.', 'wp-business-directory'));
        }

        if (email_exists($email)) {
            wp_send_json_error(__('Email address already registered.', 'wp-business-directory'));
        }

        // Create user
        $user_id = wp_create_user($username, $password, $email);

        if (is_wp_error($user_id)) {
            wp_send_json_error($user_id->get_error_message());
        }

        // Check if user is administrator (in case of direct registration)
        $user = new WP_User($user_id);
        $is_admin = in_array('administrator', $user->roles);

        // Set approval status
        update_user_meta($user_id, 'wpbd_approved', $is_admin ? 1 : 0); // Auto-approve admins

        // Set listings limit
        if ($is_admin) {
            update_user_meta($user_id, 'wpbd_listings_limit', 999); // Unlimited for admins
        } else {
            update_user_meta($user_id, 'wpbd_listings_limit', get_option('wpbd_default_listings_limit', 5));
        }

        // Notify admin of new user registration
        if (!$is_admin) {
            $admin_email = get_option('admin_email');
            $subject = __('New User Registration - Business Directory', 'wp-business-directory');
            $message = sprintf(
                __('A new user has registered for the Business Directory: %s (%s)', 'wp-business-directory'),
                $username,
                $email
            );
            wp_mail($admin_email, $subject, $message);
        }

        $message = $is_admin ?
            __('Registration successful. You are an administrator and have been automatically approved.', 'wp-business-directory') :
            __('Registration successful. Please wait for administrator approval.', 'wp-business-directory');

        wp_send_json_success($message);
    }

    /**
     * Get business data for editing
     */
    public function get_business_data()
    {
        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'], 'wpbd_nonce')) {
            wp_die('Security check failed');
        }

        // Check if user is logged in
        if (!is_user_logged_in()) {
            wp_send_json_error(__('You must be logged in to edit a business listing.', 'wp-business-directory'));
        }

        $user_id = get_current_user_id();
        $post_id = intval($_POST['post_id']);

        // Check if user owns this listing
        $post = get_post($post_id);
        if ($post->post_author != $user_id && !current_user_can('edit_others_posts')) {
            wp_send_json_error(__('You do not have permission to edit this listing.', 'wp-business-directory'));
        }

        // Get post data
        $data = array(
            'ID' => $post_id,
            'title' => $post->post_title,
            'description' => get_post_meta($post_id, '_wpbd_description', true),
            'phone' => get_post_meta($post_id, '_wpbd_phone', true),
            'address' => get_post_meta($post_id, '_wpbd_address', true),
            'website' => get_post_meta($post_id, '_wpbd_website', true),
            'categories' => wp_get_post_terms($post_id, 'wpbd_category', array('fields' => 'ids')),
        );

        wp_send_json_success($data);
    }

    /**
     * Search listings by name, NAICS code, or category
     */
    public function search_listings()
    {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'wpbd_nonce')) {
            wp_die('Security check failed');
        }

        $search_term = sanitize_text_field($_POST['search_term']);

        if (empty($search_term)) {
            wp_send_json_success(array('results' => array()));
        }

        $results = array();

        // 1. Search Categories
        $categories = get_terms(array(
            'taxonomy' => 'wpbd_category',
            'hide_empty' => false,
            'name__like' => $search_term,
            'number' => 5
        ));

        if (!is_wp_error($categories)) {
            foreach ($categories as $cat) {
                $results[] = array(
                    'id' => $cat->term_id,
                    'title' => $cat->name,
                    'type' => 'category',
                    'subtitle' => __('Category', 'wp-business-directory')
                );
            }
        }

        // 2. Search Businesses by Name
        $name_query = new WP_Query(array(
            'post_type' => 'wpbd_business',
            'post_status' => 'publish',
            'posts_per_page' => 5,
            's' => $search_term,
            'fields' => 'ids'
        ));

        if ($name_query->have_posts()) {
            foreach ($name_query->posts as $post_id) {
                $results[] = array(
                    'id' => $post_id,
                    'title' => get_the_title($post_id),
                    'type' => 'business',
                    'subtitle' => __('Business', 'wp-business-directory')
                );
            }
        }

        // 3. Search Businesses by NAICS Code
        $naics_query = new WP_Query(array(
            'post_type' => 'wpbd_business',
            'post_status' => 'publish',
            'posts_per_page' => 5,
            'meta_query' => array(
                array(
                    'key' => '_wpbd_naics_code',
                    'value' => $search_term,
                    'compare' => 'LIKE'
                )
            ),
            'fields' => 'ids'
        ));

        if ($naics_query->have_posts()) {
            foreach ($naics_query->posts as $post_id) {
                // Avoid duplicates if already found by name
                $is_duplicate = false;
                foreach ($results as $r) {
                    if ($r['type'] === 'business' && $r['id'] === $post_id) {
                        $is_duplicate = true;
                        break;
                    }
                }

                if (!$is_duplicate) {
                    $naics_code = get_post_meta($post_id, '_wpbd_naics_code', true);
                    $results[] = array(
                        'id' => $post_id,
                        'title' => get_the_title($post_id),
                        'type' => 'business',
                        'subtitle' => sprintf(__('NAICS: %s', 'wp-business-directory'), $naics_code)
                    );
                }
            }
        }


        wp_send_json_success(array(
            'results' => $results
        ));
    }

    /**
     * Get HTML for a single business listing
     */
    public function get_business_html()
    {
        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'], 'wpbd_nonce')) {
            wp_die('Security check failed');
        }

        $business_id = intval($_POST['business_id']);

        if (!$business_id) {
            wp_send_json_error('Invalid business ID');
        }

        // Get the business post
        $business = get_post($business_id);

        if (!$business || $business->post_type !== 'wpbd_business') {
            wp_send_json_error('Business not found');
        }

        // Get the HTML using the shared method
        $html = $this->get_business_listing_html($business_id);

        wp_send_json_success(array(
            'html' => $html
        ));
    }
}
?>