/**
 * Directorist Logo Field JavaScript
 */
(function($) {
    'use strict';

    var DirectoristLogoField = {
        
        /**
         * Initialize
         */
        init: function() {
            this.bindEvents();
            this.fixLogoInSlider();
            this.syncMediaUploaderFieldKey();
            this.bindFieldKeySyncEvents();
        },

        /**
         * Fix logo if it's inside a swiper/slider
         * Move it outside the slider wrapper
         */
        fixLogoInSlider: function() {
            // Wait for DOM to be ready
            $(document).ready(function() {
                // Find any logos inside swiper elements
                $('.swiper-slide .directorist-logo-wrapper, .swiper-wrapper .directorist-logo-wrapper').each(function() {
                    var $logo = $(this);
                    var $swiper = $logo.closest('.swiper, .swiper-container, .directorist-slider');
                    
                    if ($swiper.length) {
                        // Move logo before the swiper
                        $logo.detach().insertBefore($swiper);
                    }
                });
                
                // Also check for logos in slider that might not have wrapper
                $('.swiper-slide .directorist-logo-display').each(function() {
                    var $logo = $(this);
                    // Only if not already wrapped
                    if (!$logo.closest('.directorist-logo-wrapper').length) {
                        var $swiper = $logo.closest('.swiper, .swiper-container, .directorist-slider');
                        if ($swiper.length) {
                            $logo.detach().insertBefore($swiper);
                        }
                    }
                });
            });
        },

        /**
         * Sync the Directorist media uploader field key with the actual logo field key.
         * Prevents REST temp-media-upload "Invalid field" errors on submission.
         */
        syncMediaUploaderFieldKey: function() {
            try {
                if (typeof window === 'undefined' || typeof window.directorist === 'undefined') {
                    return;
                }

                var data = window.directorist.add_listing_data;
                if (!data || !Array.isArray(data.media_uploader)) {
                    return;
                }

                var $container = $('.directorist-logo-uploader').first();
                if (!$container.length) {
                    return;
                }

                var fieldKey = $container.attr('data-field-key');
                if (!fieldKey) {
                    return;
                }

                data.media_uploader.forEach(function(uploader) {
                    if (uploader && uploader.element_id === 'directorist-logo-uploader') {
                        uploader.meta_name = fieldKey;
                    }
                });
            } catch (e) {
                // Silently fail to avoid breaking the listing form.
            }
        },

        /**
         * Re-sync on listing type changes or dynamic reloads.
         */
        bindFieldKeySyncEvents: function() {
            var self = this;
            window.addEventListener('directorist-type-change', function() {
                self.syncMediaUploaderFieldKey();
            });
            window.addEventListener('directorist-reload-select2-fields', function() {
                self.syncMediaUploaderFieldKey();
            });
            document.addEventListener('directorist-reload-plupload', function() {
                self.syncMediaUploaderFieldKey();
            });
        },

        /**
         * Bind all events
         */
        bindEvents: function() {
            var self = this;

            // Click to upload (frontend - uses WP media library if available, otherwise AJAX)
            $(document).on('click', '.directorist-logo-upload-btn', function(e) {
                e.preventDefault();
                e.stopPropagation();
                
                var $uploadArea = $(this).closest('.directorist-logo-upload-area');
                
                // Try WordPress Media Library first (only if user can upload media)
                var canUploadMedia = directoristLogoField && directoristLogoField.canUploadMedia;
                if (canUploadMedia && typeof wp !== 'undefined' && typeof wp.media !== 'undefined') {
                    self.openMediaLibrary($uploadArea);
                } else {
                    self.openFileInput($uploadArea);
                }
            });

            // Click on upload area (not the button)
            $(document).on('click', '.directorist-logo-upload-area', function(e) {
                if ($(e.target).closest('.directorist-logo-preview').length || 
                    $(e.target).closest('.directorist-logo-remove').length) {
                    return;
                }
                
                $(this).find('.directorist-logo-upload-btn').trigger('click');
            });

            // Remove logo
            $(document).on('click', '.directorist-logo-remove', function(e) {
                e.preventDefault();
                e.stopPropagation();
                self.removeLogo($(this).closest('.directorist-logo-upload-area'));
            });

            // Drag and drop
            $(document)
                .on('dragover', '.directorist-logo-upload-area', function(e) {
                    e.preventDefault();
                    $(this).addClass('dragover');
                })
                .on('dragleave', '.directorist-logo-upload-area', function(e) {
                    e.preventDefault();
                    $(this).removeClass('dragover');
                })
                .on('drop', '.directorist-logo-upload-area', function(e) {
                    e.preventDefault();
                    $(this).removeClass('dragover');
                    
                    var files = e.originalEvent.dataTransfer.files;
                    if (files.length) {
                        self.handleFileUpload($(this), files[0]);
                    }
                });
        },

        /**
         * Open WordPress Media Library
         */
        openMediaLibrary: function($uploadArea) {
            var self = this;
            
            var mediaFrame = wp.media({
                title: directoristLogoField.selectImage || 'Select Logo Image',
                button: {
                    text: directoristLogoField.useImage || 'Use this image'
                },
                multiple: false,
                library: {
                    type: 'image'
                }
            });

            mediaFrame.on('select', function() {
                var attachment = mediaFrame.state().get('selection').first().toJSON();
                self.setLogo($uploadArea, attachment.id, attachment.sizes.medium ? attachment.sizes.medium.url : attachment.url);
            });

            mediaFrame.open();
        },

        /**
         * Open file input (fallback for non-authenticated users)
         */
        openFileInput: function($uploadArea) {
            var self = this;
            var $input = $('<input type="file" accept="image/jpeg,image/png,image/gif,image/webp" style="display:none;">');
            
            $input.on('change', function() {
                var file = this.files[0];
                if (file) {
                    self.handleFileUpload($uploadArea, file);
                }
                $input.remove();
            });
            
            $('body').append($input);
            $input.trigger('click');
        },

        /**
         * Handle file upload via AJAX
         */
        handleFileUpload: function($uploadArea, file) {
            var self = this;
            var maxSize = parseInt($uploadArea.data('max-size') || 2048) * 1024; // Convert to bytes

            // Validate file type
            var validTypes = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'image/webp'];
            if (validTypes.indexOf(file.type) === -1) {
                alert('Please upload a valid image file (JPG, PNG, GIF, or WebP).');
                return;
            }

            // Validate file size
            if (file.size > maxSize) {
                alert('File is too large. Maximum size is ' + ($uploadArea.data('max-size') || 2048) + ' KB.');
                return;
            }

            // Show loading state
            $uploadArea.addClass('loading');

            // Create FormData
            var formData = new FormData();
            formData.append('action', 'directorist_upload_logo');
            formData.append('nonce', directoristLogoField.nonce);
            formData.append('logo_file', file);

            // Upload via AJAX
            $.ajax({
                url: directoristLogoField.ajaxUrl,
                type: 'POST',
                data: formData,
                processData: false,
                contentType: false,
                success: function(response) {
                    $uploadArea.removeClass('loading');
                    
                    if (response.success) {
                        self.setLogo($uploadArea, response.data.id, response.data.url);
                    } else {
                        alert(response.data.message || 'Upload failed. Please try again.');
                    }
                },
                error: function() {
                    $uploadArea.removeClass('loading');
                    alert('Upload failed. Please try again.');
                }
            });
        },

        /**
         * Set logo in the field
         */
        setLogo: function($uploadArea, id, url) {
            // Update hidden input
            $uploadArea.find('input[type="hidden"]').val(id);
            
            // Update preview
            var $preview = $uploadArea.find('.directorist-logo-preview');
            $preview.find('img').attr('src', url);
            $preview.show();
            
            // Hide upload button
            $uploadArea.find('.directorist-logo-upload-btn').hide();
            
            // Add has-logo class
            $uploadArea.addClass('has-logo');
        },

        /**
         * Remove logo from the field
         */
        removeLogo: function($uploadArea) {
            // Clear hidden input
            $uploadArea.find('input[type="hidden"]').val('');
            
            // Hide preview
            $uploadArea.find('.directorist-logo-preview').hide();
            $uploadArea.find('.directorist-logo-preview img').attr('src', '');
            
            // Show upload button
            $uploadArea.find('.directorist-logo-upload-btn').show();
            
            // Remove has-logo class
            $uploadArea.removeClass('has-logo');
        }
    };

    // Initialize on document ready
    $(document).ready(function() {
        DirectoristLogoField.init();
    });

})(jQuery);

(function($){
    'use strict';

    function updateUploaderData() {
        try {
            if (typeof window === 'undefined' || typeof window.directorist === 'undefined') {
                return;
            }

            var data = window.directorist.add_listing_data;
            if (!data || !Array.isArray(data.media_uploader)) {
                return;
            }

            var $container = $('.directorist-logo-uploader').first();
            if (!$container.length) {
                return;
            }

            var fieldKey = $container.attr('data-field-key');
            if (!fieldKey) {
                return;
            }

            data.media_uploader.forEach(function(uploader){
                if (uploader && uploader.element_id === 'directorist-logo-uploader') {
                    uploader.meta_name = fieldKey;
                }
            });

            if (!window.__directoristLogoUploaderPatched && window.EzMediaUploader) {
                window.__directoristLogoUploaderPatched = true;
                var _init = window.EzMediaUploader.prototype.init;
                window.EzMediaUploader.prototype.init = function() {
                    var res = _init.apply(this, arguments);
                    try {
                        if (this.container && this.container.classList && this.container.classList.contains('directorist-logo-uploader')) {
                            window.directoristLogoUploader = this;
                        }
                    } catch (e) {}
                    return res;
                };
            }
        } catch (e) {}
    }

    function syncUploaderDataAttr() {
        var $container = $('.directorist-logo-uploader').first();
        if (!$container.length || !window.directoristLogoUploader) {
            return;
        }

        try {
            var dataAttr = $container.attr('data-uploader');
            if (!dataAttr) {
                return;
            }
            var options = JSON.parse(dataAttr);
            var merged = $.extend(true, {}, window.directoristLogoUploader.options || {}, {
                maxFileItems: options.max_num_of_img,
                maxTotalFileSize: options.max_total_img_size,
                maxFileSize: options.max_size_per_img,
                minFileItems: options.is_required,
                allowedFileFormats: options.type,
                uploaderType: options.uploader_type || 'files'
            });
            window.directoristLogoUploader.options = merged;
        } catch (e) {}
    }

    $(document).on('directorist-type-change directorist-reload-select2-fields directorist-reload-plupload', function(){
        updateUploaderData();
        syncUploaderDataAttr();
    });

    $(window).on('load', function(){
        updateUploaderData();
        syncUploaderDataAttr();
    });
})(jQuery);
