<?php
/**
 * Plugin Name: Directorist - Logo Field
 * Plugin URI: https://www.genexmarketing.com
 * Description: Adds a custom Logo image field type to Directorist that displays as an image instead of a download link.
 * Version: 1.0.0
 * Author: GenexMarketing Agency Ltd.
 * Author URI: https://www.genexmarketing.com
 * Text Domain: directorist-logo-field
 * Domain Path: /languages
 * Requires at least: 5.6
 * Requires PHP: 7.4
 * 
 * Requires Plugins: directorist
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// Plugin constants
define( 'DIRECTORIST_LOGO_FIELD_VERSION', '1.1.0' );
define( 'DIRECTORIST_LOGO_FIELD_FILE', __FILE__ );
define( 'DIRECTORIST_LOGO_FIELD_PATH', plugin_dir_path( __FILE__ ) );
define( 'DIRECTORIST_LOGO_FIELD_URL', plugin_dir_url( __FILE__ ) );

/**
 * Main plugin class
 */
final class Directorist_Logo_Field {

    /**
     * Single instance
     */
    private static $instance = null;

    /**
     * Get instance
     */
    public static function instance() {
        if ( is_null( self::$instance ) ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        $this->check_dependencies();
    }

    /**
     * Check if Directorist is active
     */
    private function check_dependencies() {
        add_action( 'plugins_loaded', array( $this, 'init' ) );
    }

    /**
     * Initialize the plugin
     */
    public function init() {
        // Always load settings (so admin can see the settings page)
        require_once DIRECTORIST_LOGO_FIELD_PATH . 'includes/class-logo-field-settings.php';
        new Directorist_Logo_Field_Settings();
        
        // Check if Directorist is active for the rest of functionality
        if ( ! class_exists( 'Directorist_Base' ) && ! post_type_exists( 'at_biz_dir' ) ) {
            add_action( 'admin_notices', array( $this, 'missing_directorist_notice' ) );
            return;
        }

        // Load includes
        $this->includes();

        // Initialize hooks
        $this->init_hooks();
    }

    /**
     * Include required files
     */
    private function includes() {
        require_once DIRECTORIST_LOGO_FIELD_PATH . 'includes/class-logo-field-type.php';
        require_once DIRECTORIST_LOGO_FIELD_PATH . 'includes/class-logo-field-frontend.php';
        require_once DIRECTORIST_LOGO_FIELD_PATH . 'includes/class-logo-field-display.php';
        require_once DIRECTORIST_LOGO_FIELD_PATH . 'includes/class-logo-field-admin.php';
        require_once DIRECTORIST_LOGO_FIELD_PATH . 'includes/class-logo-field-inject.php';
        require_once DIRECTORIST_LOGO_FIELD_PATH . 'includes/class-logo-field-directorist-field.php';
    }

    /**
     * Initialize hooks
     */
    private function init_hooks() {
        // Enqueue scripts and styles
        add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_frontend_assets' ) );
        add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_assets' ) );

        // Initialize classes
        new Directorist_Logo_Field_Type();
        new Directorist_Logo_Field_Frontend();
        new Directorist_Logo_Field_Display();
        new Directorist_Logo_Field_Admin();
        new Directorist_Logo_Field_Inject(); // Direct template injection
    }

    /**
     * Enqueue frontend assets
     */
    public function enqueue_frontend_assets() {
        // Always enqueue on frontend - Directorist forms can be on any page
        wp_enqueue_style(
            'directorist-logo-field',
            DIRECTORIST_LOGO_FIELD_URL . 'assets/css/logo-field.css',
            array(),
            DIRECTORIST_LOGO_FIELD_VERSION
        );

        wp_enqueue_script(
            'directorist-logo-field',
            DIRECTORIST_LOGO_FIELD_URL . 'assets/js/logo-field.js',
            array( 'jquery' ),
            DIRECTORIST_LOGO_FIELD_VERSION,
            true
        );

        // Enqueue media uploader only if user can access it
        $can_upload_media = is_user_logged_in() && current_user_can( 'upload_files' );
        if ( $can_upload_media ) {
            wp_enqueue_media();
        }

        wp_localize_script( 'directorist-logo-field', 'directoristLogoField', array(
            'ajaxUrl' => admin_url( 'admin-ajax.php' ),
            'nonce' => wp_create_nonce( 'directorist_logo_field_nonce' ),
            'selectImage' => __( 'Select Logo Image', 'directorist-logo-field' ),
            'useImage' => __( 'Use this image', 'directorist-logo-field' ),
            'removeImage' => __( 'Remove', 'directorist-logo-field' ),
            'canUploadMedia' => $can_upload_media,
        ) );
    }

    /**
     * Enqueue admin assets
     */
    public function enqueue_admin_assets( $hook ) {
        global $post_type, $pagenow;

        // Load on listing edit screens
        $is_listing_edit = ( 'post.php' === $hook || 'post-new.php' === $hook ) && 'at_biz_dir' === $post_type;
        
        // Also load on Directorist settings pages (Directory Builder)
        $is_directorist_page = isset( $_GET['page'] ) && strpos( $_GET['page'], 'atbdp' ) !== false;
        
        if ( $is_listing_edit || $is_directorist_page ) {
            wp_enqueue_media();

            wp_enqueue_style(
                'directorist-logo-field-admin',
                DIRECTORIST_LOGO_FIELD_URL . 'assets/css/logo-field.css',
                array(),
                DIRECTORIST_LOGO_FIELD_VERSION
            );

            wp_enqueue_script(
                'directorist-logo-field-admin',
                DIRECTORIST_LOGO_FIELD_URL . 'assets/js/logo-field.js',
                array( 'jquery' ),
                DIRECTORIST_LOGO_FIELD_VERSION,
                true
            );

            $can_upload_media = current_user_can( 'upload_files' );
            wp_localize_script( 'directorist-logo-field-admin', 'directoristLogoField', array(
                'ajaxUrl' => admin_url( 'admin-ajax.php' ),
                'nonce' => wp_create_nonce( 'directorist_logo_field_nonce' ),
                'selectImage' => __( 'Select Logo Image', 'directorist-logo-field' ),
                'useImage' => __( 'Use this image', 'directorist-logo-field' ),
                'removeImage' => __( 'Remove', 'directorist-logo-field' ),
                'canUploadMedia' => $can_upload_media,
            ) );
        }
    }

    /**
     * Admin notice for missing Directorist
     */
    public function missing_directorist_notice() {
        ?>
        <div class="notice notice-error">
            <p>
                <?php
                printf(
                    esc_html__( '%1$s requires %2$s to be installed and activated.', 'directorist-logo-field' ),
                    '<strong>Directorist Logo Field</strong>',
                    '<strong>Directorist</strong>'
                );
                ?>
            </p>
        </div>
        <?php
    }
}

/**
 * Initialize the plugin
 */
function directorist_logo_field() {
    return Directorist_Logo_Field::instance();
}

// Start the plugin
directorist_logo_field();

// Activation hook - set defaults
register_activation_hook( __FILE__, 'directorist_logo_field_activate' );
function directorist_logo_field_activate() {
    // Always set/update default options on activation
    $defaults = array(
        'show_on_single' => 1,
        'single_width' => 180,
        'show_on_listings' => 1,
        'listings_width' => 100,
    );
    
    $existing = get_option( 'directorist_logo_field_options', array() );
    $merged = wp_parse_args( $existing, $defaults );
    update_option( 'directorist_logo_field_options', $merged );
    
    // Flush rewrite rules
    flush_rewrite_rules();
}

// Also set defaults if options don't exist (for manual uploads)
add_action( 'admin_init', 'directorist_logo_field_maybe_set_defaults' );
function directorist_logo_field_maybe_set_defaults() {
    if ( false === get_option( 'directorist_logo_field_options' ) ) {
        $defaults = array(
            'show_on_single' => 1,
            'single_width' => 180,
            'show_on_listings' => 1,
            'listings_width' => 100,
        );
        add_option( 'directorist_logo_field_options', $defaults );
    }
}
