<?php
/**
 * Logo Field Admin Handler
 * 
 * Handles the logo field in WordPress admin backend
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class Directorist_Logo_Field_Admin {

    /**
     * Constructor
     */
    public function __construct() {
        $this->init_hooks();
    }

    /**
     * Initialize hooks
     */
    private function init_hooks() {
        // Add meta box to listing edit screen
        add_action( 'add_meta_boxes', array( $this, 'add_logo_meta_box' ) );
        
        // Save meta box data
        add_action( 'save_post_at_biz_dir', array( $this, 'save_logo_meta_box' ), 10, 2 );
    }

    /**
     * Add logo meta box
     */
    public function add_logo_meta_box() {
        add_meta_box(
            'directorist_logo_meta_box',
            __( 'Listing Logo', 'directorist-logo-field' ),
            array( $this, 'render_logo_meta_box' ),
            'at_biz_dir',
            'side',
            'default'
        );
    }

    /**
     * Render logo meta box
     */
    public function render_logo_meta_box( $post ) {
        // Add nonce for security
        wp_nonce_field( 'directorist_logo_meta_box', 'directorist_logo_meta_box_nonce' );
        
        $logo_id = get_post_meta( $post->ID, '_directorist_logo', true );
        $logo_url = $logo_id ? wp_get_attachment_image_url( $logo_id, 'medium' ) : '';
        $has_logo = ! empty( $logo_url );
        ?>
        <div class="directorist-logo-admin-wrap">
            <div class="directorist-logo-upload-area <?php echo $has_logo ? 'has-logo' : ''; ?>">
                <input type="hidden" 
                       name="directorist_logo" 
                       id="directorist_logo_id" 
                       value="<?php echo esc_attr( $logo_id ); ?>" />
                
                <div class="directorist-logo-preview" <?php echo ! $has_logo ? 'style="display:none;"' : ''; ?>>
                    <img src="<?php echo esc_url( $logo_url ); ?>" alt="Logo" style="max-width: 100%; height: auto;" />
                    <p>
                        <button type="button" class="button directorist-logo-remove">
                            <?php esc_html_e( 'Remove Logo', 'directorist-logo-field' ); ?>
                        </button>
                    </p>
                </div>
                
                <div class="directorist-logo-upload-btn-wrap" <?php echo $has_logo ? 'style="display:none;"' : ''; ?>>
                    <button type="button" class="button button-primary directorist-logo-upload-btn">
                        <?php esc_html_e( 'Upload Logo', 'directorist-logo-field' ); ?>
                    </button>
                    <p class="description">
                        <?php esc_html_e( 'Upload a logo image for this listing.', 'directorist-logo-field' ); ?>
                    </p>
                </div>
            </div>
        </div>
        
        <style>
            .directorist-logo-admin-wrap { padding: 10px 0; }
            .directorist-logo-preview img { 
                border: 1px solid #ddd; 
                padding: 5px; 
                background: #fff;
                border-radius: 4px;
            }
            .directorist-logo-preview p { margin-top: 10px; }
            .directorist-logo-upload-btn-wrap .description { margin-top: 8px; }
        </style>
        
        <script type="text/javascript">
        jQuery(document).ready(function($) {
            var logoFrame;
            
            // Upload button click
            $('#directorist_logo_meta_box').on('click', '.directorist-logo-upload-btn', function(e) {
                e.preventDefault();
                
                if (logoFrame) {
                    logoFrame.open();
                    return;
                }
                
                logoFrame = wp.media({
                    title: '<?php echo esc_js( __( 'Select Logo Image', 'directorist-logo-field' ) ); ?>',
                    button: { text: '<?php echo esc_js( __( 'Use this image', 'directorist-logo-field' ) ); ?>' },
                    multiple: false,
                    library: { type: 'image' }
                });
                
                logoFrame.on('select', function() {
                    var attachment = logoFrame.state().get('selection').first().toJSON();
                    var imgUrl = attachment.sizes && attachment.sizes.medium ? attachment.sizes.medium.url : attachment.url;
                    
                    $('#directorist_logo_id').val(attachment.id);
                    $('#directorist_logo_meta_box .directorist-logo-preview img').attr('src', imgUrl);
                    $('#directorist_logo_meta_box .directorist-logo-preview').show();
                    $('#directorist_logo_meta_box .directorist-logo-upload-btn-wrap').hide();
                });
                
                logoFrame.open();
            });
            
            // Remove button click
            $('#directorist_logo_meta_box').on('click', '.directorist-logo-remove', function(e) {
                e.preventDefault();
                $('#directorist_logo_id').val('');
                $('#directorist_logo_meta_box .directorist-logo-preview').hide();
                $('#directorist_logo_meta_box .directorist-logo-upload-btn-wrap').show();
            });
        });
        </script>
        <?php
    }

    /**
     * Save logo meta box
     */
    public function save_logo_meta_box( $post_id, $post ) {
        // Check nonce
        if ( ! isset( $_POST['directorist_logo_meta_box_nonce'] ) || 
             ! wp_verify_nonce( $_POST['directorist_logo_meta_box_nonce'], 'directorist_logo_meta_box' ) ) {
            return;
        }

        // Check autosave
        if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
            return;
        }

        // Check permissions
        if ( ! current_user_can( 'edit_post', $post_id ) ) {
            return;
        }

        // Save or delete the logo
        if ( isset( $_POST['directorist_logo'] ) ) {
            $logo_id = absint( $_POST['directorist_logo'] );
            
            if ( $logo_id && wp_attachment_is_image( $logo_id ) ) {
                update_post_meta( $post_id, '_directorist_logo', $logo_id );
            } else {
                delete_post_meta( $post_id, '_directorist_logo' );
            }
        }
    }
}
