<?php
/**
 * Directorist Logo Field - Field Class
 *
 * Handles logo uploads in frontend listing submission.
 */

namespace Directorist\Fields;

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

if ( ! class_exists( Base_Field::class ) || ! class_exists( Fields::class ) ) {
    return;
}

class Logo_Upload_Field extends Base_Field {
    public $type = 'logo';

    public function get_value( $posted_data ) {
        if ( empty( $posted_data[ $this->get_key() ] ) && empty( $posted_data[ $this->get_key() . '_old' ] ) ) {
            return null;
        }

        $new_images = (array) directorist_get_var( $posted_data[ $this->get_key() ], [] );
        $old_images = (array) directorist_get_var( $posted_data[ $this->get_key() . '_old' ], [] );

        $maybe_old_images = array_filter( $new_images, 'is_numeric' );

        if ( count( $maybe_old_images ) > 0 ) {
            $old_images = array_merge( $old_images, $maybe_old_images );
            $new_images = array_diff( $new_images, $maybe_old_images );
        }

        return array(
            'new' => array_filter( $new_images ),
            'old' => array_filter( wp_parse_id_list( $old_images ) ),
        );
    }

    public function validate( $posted_data ) {
        $files = $this->get_value( $posted_data );
        if ( is_null( $files ) ) {
            if ( $this->is_required() ) {
                $this->add_error( __( 'This field is required.', 'directorist' ) );
                return false;
            }
            return true;
        }

        $old_images = $files['old'];
        $new_images = $files['new'];

        if ( $this->is_required() && empty( $old_images ) && empty( $new_images ) ) {
            $this->add_error( __( 'This field is required.', 'directorist' ) );
            return false;
        }

        if ( ( count( $old_images ) + count( $new_images ) ) > 1 ) {
            $this->add_error( __( 'Only one logo image is allowed.', 'directorist' ) );
            return false;
        }

        $max_size_kb = isset( $this->max_file_size ) ? absint( $this->max_file_size ) : 2048;
        $max_size    = $max_size_kb * 1024;

        $upload_dir = wp_get_upload_dir();
        $temp_dir   = trailingslashit( $upload_dir['basedir'] ) . trailingslashit( directorist_get_temp_upload_dir() . DIRECTORY_SEPARATOR . date( 'nj' ) );

        foreach ( $new_images as $file ) {
            $filepath = realpath( $temp_dir . $file );

            if ( empty( $file ) || ! $filepath ) {
                continue;
            }

            $filesize  = filesize( $filepath );
            $real_mime = wp_get_image_mime( $filepath );

            if ( ! $real_mime || strpos( $real_mime, 'image' ) === false ) {
                $this->add_error(
                    sprintf(
                        __( '[%1$s] invalid file type, only image allowed.', 'directorist' ),
                        $file
                    )
                );
                continue;
            }

            if ( $filesize > $max_size ) {
                $this->add_error(
                    sprintf(
                        __( '[%1$s] size exceeded, %2$s is allowed only.', 'directorist' ),
                        $file,
                        size_format( $max_size )
                    )
                );
            }
        }

        return ! $this->has_error();
    }
}

Fields::register( new Logo_Upload_Field() );
