<?php
/**
 * Logo Field Display Handler
 * 
 * Handles displaying the logo in single listing and all listings layouts
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class Directorist_Logo_Field_Display {

    /**
     * Constructor
     */
    public function __construct() {
        $this->init_hooks();
    }

    /**
     * Initialize hooks
     */
    private function init_hooks() {
        // ========================================
        // SINGLE LISTING PAGE - Widget Rendering (Actions)
        // ========================================
        
        // Primary widget action hooks - Directorist 7.x pattern
        add_action( 'atbdp_widget_logo', array( $this, 'render_single_widget' ), 10, 4 );
        add_action( 'directorist_single_listing_widget_logo', array( $this, 'render_single_widget' ), 10, 4 );
        add_action( 'directorist_single_widget_logo', array( $this, 'render_single_widget' ), 10, 4 );
        
        // Section-specific widget hooks
        add_action( 'atbdp_single_listing_content_widget_logo', array( $this, 'render_single_widget' ), 10, 4 );
        add_action( 'atbdp_single_listing_header_widget_logo', array( $this, 'render_single_widget' ), 10, 4 );
        add_action( 'atbdp_single_listing_quick_info_widget_logo', array( $this, 'render_single_widget' ), 10, 4 );
        add_action( 'directorist_single_listing_content_widget_logo', array( $this, 'render_single_widget' ), 10, 4 );
        add_action( 'directorist_single_listing_header_widget_logo', array( $this, 'render_single_widget' ), 10, 4 );
        
        // Field-specific hooks
        add_action( 'atbdp_single_listing_field_logo', array( $this, 'render_single_field' ), 10, 2 );
        add_action( 'directorist_single_listing_logo', array( $this, 'render_current_listing' ), 10 );
        
        // Generic widget rendering - this is the most common pattern
        add_action( 'atbdp_single_listing_logo', array( $this, 'render_current_listing' ), 10 );
        add_action( 'directorist_widget_logo', array( $this, 'render_single_widget' ), 10, 4 );
        
        // ========================================
        // ALL LISTINGS / CARD - Widget Rendering
        // ========================================
        
        // Card widget action hooks
        add_action( 'atbdp_listings_card_widget_logo', array( $this, 'render_card_widget' ), 10, 4 );
        add_action( 'directorist_listing_card_widget_logo', array( $this, 'render_card_widget' ), 10, 4 );
        add_action( 'atbdp_listings_card_field_logo', array( $this, 'render_card_field' ), 10, 2 );
        add_action( 'directorist_all_listings_card_widget_logo', array( $this, 'render_card_widget' ), 10, 4 );
        
        // Card section hooks
        add_action( 'atbdp_listing_card_header_widget_logo', array( $this, 'render_card_widget' ), 10, 4 );
        add_action( 'atbdp_listing_card_body_widget_logo', array( $this, 'render_card_widget' ), 10, 4 );
        add_action( 'atbdp_listing_card_footer_widget_logo', array( $this, 'render_card_widget' ), 10, 4 );
        add_action( 'directorist_listing_card_header_widget_logo', array( $this, 'render_card_widget' ), 10, 4 );
        add_action( 'directorist_listing_card_body_widget_logo', array( $this, 'render_card_widget' ), 10, 4 );
        
        // Grid and list view
        add_action( 'atbdp_grid_view_widget_logo', array( $this, 'render_card_widget' ), 10, 4 );
        add_action( 'atbdp_list_view_widget_logo', array( $this, 'render_card_widget' ), 10, 4 );
        
        // ========================================
        // FILTER HOOKS (return HTML)
        // ========================================
        
        add_filter( 'directorist_single_field_logo', array( $this, 'filter_single_listing' ), 10, 3 );
        add_filter( 'directorist_single_listing_widget_logo_html', array( $this, 'filter_widget_html' ), 10, 4 );
        add_filter( 'atbdp_custom_field_value_logo', array( $this, 'filter_custom_field_value' ), 10, 3 );
        add_filter( 'directorist_listing_card_field_logo_html', array( $this, 'filter_card_html' ), 10, 3 );
        
        // ========================================
        // TEMPLATE INTEGRATION
        // ========================================
        
        // Hook into the widget template system
        add_filter( 'directorist_widget_template_logo', array( $this, 'get_widget_template' ), 10, 2 );
        add_filter( 'atbdp_widget_template_logo', array( $this, 'get_widget_template' ), 10, 2 );
        
        // Include our widget in widget rendering loop
        add_action( 'directorist_before_single_listing_section', array( $this, 'maybe_render_in_section' ), 10, 3 );
        
        // ========================================
        // SHORTCODE
        // ========================================
        
        add_shortcode( 'directorist_logo', array( $this, 'logo_shortcode' ) );
        
        // Note: Direct card/single injection is now handled by class-logo-field-inject.php
        // This class only handles builder widget rendering for compatibility
    }

    /**
     * Render widget in single listing page (action hook)
     */
    public function render_single_widget( $widget_data = array(), $listing_id = 0, $listing = null, $section_data = array() ) {
        // Try to get listing ID from various sources
        if ( ! $listing_id && $listing ) {
            $listing_id = is_object( $listing ) ? $listing->ID : absint( $listing );
        }
        if ( ! $listing_id ) {
            global $post;
            if ( $post && 'at_biz_dir' === $post->post_type ) {
                $listing_id = $post->ID;
            }
        }
        
        if ( ! $listing_id ) {
            return;
        }

        echo $this->get_logo_html( $listing_id, $widget_data, 'single' );
    }

    /**
     * Render single field (action hook)
     */
    public function render_single_field( $field_data, $listing_id ) {
        if ( ! $listing_id ) {
            global $post;
            $listing_id = $post ? $post->ID : 0;
        }
        
        echo $this->get_logo_html( $listing_id, $field_data, 'single' );
    }

    /**
     * Render for current listing (no params)
     */
    public function render_current_listing() {
        global $post;
        
        if ( ! $post || 'at_biz_dir' !== $post->post_type ) {
            return;
        }
        
        echo $this->get_logo_html( $post->ID, array(), 'single' );
    }

    /**
     * Render widget in card/grid layout (action hook)
     */
    public function render_card_widget( $widget_data = array(), $listing_id = 0, $listing = null, $section_data = array() ) {
        if ( ! $listing_id && $listing ) {
            $listing_id = is_object( $listing ) ? $listing->ID : absint( $listing );
        }
        if ( ! $listing_id ) {
            global $post;
            if ( $post ) {
                $listing_id = $post->ID;
            }
        }
        
        if ( ! $listing_id ) {
            return;
        }

        echo $this->get_logo_html( $listing_id, $widget_data, 'card' );
    }

    /**
     * Render card field (action hook)
     */
    public function render_card_field( $field_data, $listing_id ) {
        echo $this->get_logo_html( $listing_id, $field_data, 'card' );
    }

    /**
     * Filter: Return HTML for single listing
     */
    public function filter_single_listing( $output, $field, $listing = null ) {
        $listing_id = 0;
        
        if ( $listing ) {
            $listing_id = is_object( $listing ) ? $listing->ID : absint( $listing );
        }
        
        if ( ! $listing_id ) {
            global $post;
            $listing_id = $post ? $post->ID : 0;
        }
        
        return $this->get_logo_html( $listing_id, $field, 'single' );
    }

    /**
     * Filter: Return widget HTML
     */
    public function filter_widget_html( $output, $widget_data, $listing, $section_data = array() ) {
        $listing_id = $listing ? ( is_object( $listing ) ? $listing->ID : absint( $listing ) ) : 0;
        
        if ( ! $listing_id ) {
            global $post;
            $listing_id = $post ? $post->ID : 0;
        }
        
        return $this->get_logo_html( $listing_id, $widget_data, 'single' );
    }

    /**
     * Filter: Custom field value
     */
    public function filter_custom_field_value( $value, $field, $listing_id ) {
        return $this->get_logo_html( $listing_id, $field, 'single' );
    }

    /**
     * Filter: Card HTML
     */
    public function filter_card_html( $output, $field, $listing = null ) {
        $listing_id = $listing ? ( is_object( $listing ) ? $listing->ID : absint( $listing ) ) : 0;
        
        if ( ! $listing_id ) {
            global $post;
            $listing_id = $post ? $post->ID : 0;
        }
        
        return $this->get_logo_html( $listing_id, $field, 'card' );
    }

    /**
     * Get widget template path
     */
    public function get_widget_template( $template, $widget_data = array() ) {
        return 'logo-display';
    }

    /**
     * Maybe render logo in section if it's in the section widgets
     */
    public function maybe_render_in_section( $section_data, $listing, $widgets = array() ) {
        if ( ! is_array( $widgets ) ) {
            return;
        }
        
        foreach ( $widgets as $widget ) {
            $widget_name = '';
            
            if ( is_array( $widget ) ) {
                $widget_name = isset( $widget['widget_name'] ) ? $widget['widget_name'] : '';
                if ( ! $widget_name ) {
                    $widget_name = isset( $widget['type'] ) ? $widget['type'] : '';
                }
            } elseif ( is_string( $widget ) ) {
                $widget_name = $widget;
            }
            
            if ( 'logo' === $widget_name ) {
                $listing_id = is_object( $listing ) ? $listing->ID : absint( $listing );
                echo $this->get_logo_html( $listing_id, is_array( $widget ) ? $widget : array(), 'single' );
            }
        }
    }

    /**
     * Get logo HTML
     */
    public function get_logo_html( $listing_id, $field_data = array(), $context = 'single' ) {
        if ( ! $listing_id ) {
            return '';
        }

        if ( 'card' === $context && ! $this->should_show_logo_in_cards() ) {
            return '';
        }

        $logo_id = get_post_meta( $listing_id, '_directorist_logo', true );
        
        if ( ! $logo_id ) {
            return '';
        }

        // Get display settings from field data or widget data
        // Default widths: 200px for single listing, 120px for card
        $width = 'card' === $context ? 120 : 200;
        
        if ( isset( $field_data['display_width'] ) && intval( $field_data['display_width'] ) > 0 ) {
            $width = intval( $field_data['display_width'] );
        } elseif ( isset( $field_data['options']['display_width']['value'] ) && intval( $field_data['options']['display_width']['value'] ) > 0 ) {
            $width = intval( $field_data['options']['display_width']['value'] );
        }
        
        $label = __( 'Logo', 'directorist-logo-field' );
        if ( isset( $field_data['label'] ) ) {
            $label = $field_data['label'];
        } elseif ( isset( $field_data['options']['label']['value'] ) ) {
            $label = $field_data['options']['label']['value'];
        }
        
        $show_label = false;
        if ( isset( $field_data['show_label'] ) ) {
            $show_label = $field_data['show_label'];
        } elseif ( isset( $field_data['options']['show_label']['value'] ) ) {
            $show_label = $field_data['options']['show_label']['value'];
        }

        // Get image URL - use medium size for good quality
        $logo_url = wp_get_attachment_image_url( $logo_id, 'medium' );
        
        if ( ! $logo_url ) {
            return '';
        }

        // Get listing title for alt text
        $listing_title = get_the_title( $listing_id );
        $alt_text = sprintf( __( '%s logo', 'directorist-logo-field' ), $listing_title );

        // Build style - only width, height is auto for aspect ratio
        $img_style = 'width: ' . $width . 'px; max-width: 100%; height: auto;';

        $class = 'single' === $context ? 'directorist-single-logo' : 'directorist-card-logo';
        $wrapper_class = 'directorist-logo-wrapper';
        if ( 'card' === $context ) {
            $wrapper_class .= ' directorist-logo-wrapper--card';
        } else {
            $wrapper_class .= ' directorist-logo-wrapper--single';
        }

        ob_start();
        ?>
        <div class="<?php echo esc_attr( $wrapper_class ); ?>" style="display: block; position: relative; padding: 0; margin: 0;">
            <div class="directorist-logo-display <?php echo esc_attr( $class ); ?>" data-logo-widget="true">
                <?php if ( $show_label && $label ) : ?>
                    <span class="directorist-logo-label"><?php echo esc_html( $label ); ?>:</span>
                <?php endif; ?>
                <?php if ( 'card' === $context ) : ?>
                    <a class="directorist-logo-link" href="<?php echo esc_url( get_permalink( $listing_id ) ); ?>">
                        <img src="<?php echo esc_url( $logo_url ); ?>" 
                             alt="<?php echo esc_attr( $alt_text ); ?>" 
                             class="directorist-logo-image"
                             style="<?php echo esc_attr( $img_style ); ?>"
                             loading="lazy" />
                    </a>
                <?php else : ?>
                    <img src="<?php echo esc_url( $logo_url ); ?>" 
                         alt="<?php echo esc_attr( $alt_text ); ?>" 
                         class="directorist-logo-image"
                         style="<?php echo esc_attr( $img_style ); ?>"
                         loading="lazy" />
                <?php endif; ?>
            </div>
        </div>
        <?php
        return ob_get_clean();
    }

    /**
     * Logo shortcode
     * Usage: [directorist_logo id="123" width="200"]
     */
    public function logo_shortcode( $atts ) {
        $atts = shortcode_atts( array(
            'id'     => 0,
            'width'  => 180,
        ), $atts, 'directorist_logo' );

        $listing_id = absint( $atts['id'] );
        
        if ( ! $listing_id ) {
            global $post;
            $listing_id = $post ? $post->ID : 0;
        }

        if ( ! $listing_id ) {
            return '';
        }

        $field_data = array(
            'display_width' => absint( $atts['width'] ),
        );

        return $this->get_logo_html( $listing_id, $field_data );
    }

    /**
     * Check if logo should be shown in cards (option-based)
     */
    private function should_show_logo_in_cards() {
        $enabled = Directorist_Logo_Field_Settings::get_option( 'show_on_listings', 1 );
        return apply_filters( 'directorist_logo_field_show_in_cards', (bool) $enabled );
    }

    /**
     * Inject logo before listing title in cards
     */
    public function inject_logo_before_title( $listing_id = 0 ) {
        if ( ! $this->should_show_logo_in_cards() ) {
            return;
        }
        
        if ( ! $listing_id ) {
            global $post;
            $listing_id = $post ? $post->ID : 0;
        }
        
        if ( ! $listing_id ) {
            return;
        }
        
        // Only show if listing has a logo
        $logo_id = get_post_meta( $listing_id, '_directorist_logo', true );
        if ( ! $logo_id ) {
            return;
        }
        
        $field_data = array(
            'display_width' => 100,
        );
        
        echo $this->get_logo_html( $listing_id, $field_data, 'card' );
    }

    /**
     * Maybe inject logo in card (conditional)
     */
    public function maybe_inject_logo_in_card( $listing_id = 0 ) {
        if ( ! $this->should_show_logo_in_cards() ) {
            return;
        }
        
        if ( ! $listing_id ) {
            global $post;
            $listing_id = $post ? $post->ID : 0;
        }
        
        if ( ! $listing_id ) {
            return;
        }
        
        // Only show if listing has a logo
        $logo_id = get_post_meta( $listing_id, '_directorist_logo', true );
        if ( ! $logo_id ) {
            return;
        }
        
        $field_data = array(
            'display_width' => 100,
        );
        
        echo $this->get_logo_html( $listing_id, $field_data, 'card' );
    }
}
