<?php
/**
 * Logo Field Frontend Handler
 * 
 * Handles the logo upload form field rendering and saving
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class Directorist_Logo_Field_Frontend {

    /**
     * Constructor
     */
    public function __construct() {
        $this->init_hooks();
    }

    /**
     * Initialize hooks
     */
    private function init_hooks() {
        // Register the field template for Directorist form builder
        add_filter( 'directorist_field_template', array( $this, 'register_field_template' ), 10, 2 );
        add_filter( 'directorist_template_file_path', array( $this, 'filter_template_file_path' ), 10, 3 );
        
        // Hook into form field rendering - multiple hooks for different Directorist versions
        add_action( 'directorist_listing_form_custom_field_logo', array( $this, 'render_form_field' ), 10, 2 );
        add_action( 'atbdp_add_listing_form_field_logo', array( $this, 'render_form_field' ), 10, 2 );
        add_action( 'atbdp_edit_listing_form_field_logo', array( $this, 'render_form_field' ), 10, 2 );
        add_action( 'directorist_add_listing_form_field_logo', array( $this, 'render_form_field' ), 10, 2 );
        add_action( 'directorist_edit_listing_form_field_logo', array( $this, 'render_form_field' ), 10, 2 );
        add_action( 'directorist_dashboard_add_listing_form_field_logo', array( $this, 'render_form_field' ), 10, 2 );
        add_action( 'directorist_dashboard_edit_listing_form_field_logo', array( $this, 'render_form_field' ), 10, 2 );
        
        // Hook into the listing form field output
        add_filter( 'directorist_listing_form_field_html', array( $this, 'filter_field_html' ), 10, 3 );

        // Register as a known widget type  
        add_filter( 'directorist_form_field_types', array( $this, 'add_form_field_type' ) );

        // Register with Directorist media uploader (EzMediaUploader)
        add_filter( 'atbdp_media_uploader', array( $this, 'register_media_uploader' ), 20, 1 );
        
        // Template override for the field
        add_filter( 'atbdp_listing_form_field_template_logo', array( $this, 'get_field_template' ), 10, 2 );
        add_filter( 'directorist_listing_form_field_template_logo', array( $this, 'get_field_template' ), 10, 2 );
        
        // Save the field value - multiple hooks for compatibility
        add_action( 'atbdp_listing_inserted', array( $this, 'save_logo_field' ), 10, 3 );
        add_action( 'atbdp_listing_updated', array( $this, 'save_logo_field' ), 10, 3 );
        add_action( 'directorist_listing_form_submit', array( $this, 'save_on_submit' ), 10, 2 );
        add_action( 'save_post_at_biz_dir', array( $this, 'save_on_post_save' ), 10, 3 );

        // Process temp uploads from frontend submission
        add_action( 'atbdp_after_created_listing', array( $this, 'handle_logo_uploads' ), 10, 1 );
        add_action( 'atbdp_listing_updated', array( $this, 'handle_logo_uploads' ), 10, 1 );
        
        // AJAX handlers for image upload
        add_action( 'wp_ajax_directorist_upload_logo', array( $this, 'ajax_upload_logo' ) );
        add_action( 'wp_ajax_nopriv_directorist_upload_logo', array( $this, 'ajax_upload_logo' ) );

        // Ensure REST temp upload accepts logo field
        add_action( 'rest_api_init', array( $this, 'register_rest_upload_fix' ) );

        // Ensure logo field exists on all directory types
        add_action( 'init', array( $this, 'ensure_logo_field_for_all_directories' ), 20 );
    }

    /**
     * Add our field type to recognized form field types
     */
    public function add_form_field_type( $types ) {
        $types['logo'] = __( 'Logo Image', 'directorist-logo-field' );
        return $types;
    }

    /**
     * Register media uploader config for frontend (EzMediaUploader)
     */
    public function register_media_uploader( $uploaders ) {
        if ( ! is_array( $uploaders ) ) {
            $uploaders = array();
        }

        $this->ensure_logo_field_key();

        $element_id = 'directorist-logo-uploader';
        $field_key  = $this->get_logo_field_key();
        foreach ( $uploaders as $uploader ) {
            if ( isset( $uploader['element_id'] ) && $uploader['element_id'] === $element_id ) {
                return $uploaders;
            }
        }

        $uploaders[] = array(
            'element_id'      => $element_id,
            'meta_name'       => $field_key,
            'files_meta_name' => 'directorist_logo_files_meta',
            'error_msg'       => __( 'Logo has invalid file.', 'directorist-logo-field' ),
        );

        return $uploaders;
    }

    private function get_logo_field_key() {
        $default = 'directorist_logo';
        $directory_id = $this->get_current_directory_id();

        if ( $directory_id && function_exists( 'directorist_get_listing_form_fields' ) ) {
            $fields = directorist_get_listing_form_fields( $directory_id );
            foreach ( $fields as $field ) {
                $widget_name = isset( $field['widget_name'] ) ? $field['widget_name'] : '';
                $type = isset( $field['type'] ) ? $field['type'] : '';
                if ( 'logo' === $widget_name || 'logo' === $type ) {
                    $key = isset( $field['field_key'] ) ? $field['field_key'] : '';
                    if ( $key ) {
                        return $key;
                    }
                }
            }

            $this->ensure_logo_field_present( $directory_id );
        }

        return $default;
    }

    private function ensure_logo_field_key( $directory_id = 0 ) {
        if ( ! $directory_id ) {
            $directory_id = $this->get_current_directory_id();
        }

        if ( ! $directory_id ) {
            return;
        }

        $form_data = get_term_meta( $directory_id, 'submission_form_fields', true );
        if ( empty( $form_data ) || empty( $form_data['fields'] ) || ! is_array( $form_data['fields'] ) ) {
            return;
        }

        $changed = false;
        foreach ( $form_data['fields'] as $key => $field ) {
            if ( ! is_array( $field ) ) {
                continue;
            }

            $field_type = isset( $field['widget_name'] ) ? $field['widget_name'] : ( $field['type'] ?? '' );
            if ( 'logo' !== $field_type ) {
                continue;
            }

            if ( empty( $field['field_key'] ) ) {
                $field['field_key'] = 'directorist_logo';
                $form_data['fields'][ $key ] = $field;
                $changed = true;
            }
        }

        if ( $changed ) {
            update_term_meta( $directory_id, 'submission_form_fields', $form_data );
        }
    }

    private function ensure_logo_field_present( $directory_id ) {
        $form_data = get_term_meta( $directory_id, 'submission_form_fields', true );
        if ( empty( $form_data ) || empty( $form_data['fields'] ) || ! is_array( $form_data['fields'] ) ) {
            return;
        }

        foreach ( $form_data['fields'] as $field ) {
            if ( ! is_array( $field ) ) {
                continue;
            }

            $field_type = isset( $field['widget_name'] ) ? $field['widget_name'] : ( $field['type'] ?? '' );
            if ( 'logo' === $field_type ) {
                return;
            }
        }

        $field_id = 'logo';
        if ( isset( $form_data['fields'][ $field_id ] ) ) {
            $field_id = 'logo_' . time();
        }

        $form_data['fields'][ $field_id ] = array(
            'type'        => 'logo',
            'widget_name' => 'logo',
            'widget_key'  => 'logo',
            'field_key'   => 'directorist_logo',
            'label'       => __( 'Logo', 'directorist-logo-field' ),
            'placeholder' => __( 'Click to upload logo', 'directorist-logo-field' ),
            'required'    => false,
            'only_for_admin' => false,
            'max_file_size'  => 2048,
        );

        if ( empty( $form_data['groups'] ) || ! is_array( $form_data['groups'] ) ) {
            $form_data['groups'] = array(
                array(
                    'label'  => __( 'General Information', 'directorist' ),
                    'fields' => array( $field_id ),
                ),
            );
        } else {
            $group_index = 0;
            if ( empty( $form_data['groups'][ $group_index ]['fields'] ) || ! is_array( $form_data['groups'][ $group_index ]['fields'] ) ) {
                $form_data['groups'][ $group_index ]['fields'] = array();
            }

            if ( ! in_array( $field_id, $form_data['groups'][ $group_index ]['fields'], true ) ) {
                $form_data['groups'][ $group_index ]['fields'][] = $field_id;
            }
        }

        update_term_meta( $directory_id, 'submission_form_fields', $form_data );
    }

    public function ensure_logo_field_for_all_directories() {
        $taxonomy = defined( 'ATBDP_DIRECTORY_TYPE' ) ? ATBDP_DIRECTORY_TYPE : ( defined( 'ATBDP_TYPE' ) ? ATBDP_TYPE : 'atbdp_listing_types' );
        if ( ! taxonomy_exists( $taxonomy ) ) {
            return;
        }

        $terms = get_terms(
            array(
                'taxonomy'   => $taxonomy,
                'hide_empty' => false,
                'fields'     => 'ids',
            )
        );

        if ( is_wp_error( $terms ) || empty( $terms ) ) {
            return;
        }

        foreach ( $terms as $term_id ) {
            $this->ensure_logo_field_present( absint( $term_id ) );
            $this->ensure_logo_field_key_from_directory( absint( $term_id ) );
        }
    }

    private function get_current_directory_id() {
        $listing_id = 0;

        if ( isset( $_REQUEST['listing_id'] ) ) {
            $listing_id = absint( $_REQUEST['listing_id'] );
        } elseif ( isset( $_REQUEST['edit'] ) ) {
            $listing_id = absint( $_REQUEST['edit'] );
        }

        if ( $listing_id && function_exists( 'directorist_get_listing_directory' ) ) {
            $directory_id = directorist_get_listing_directory( $listing_id );
            if ( $directory_id ) {
                return absint( $directory_id );
            }
        }

        $directory = isset( $_REQUEST['directory_type'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['directory_type'] ) ) : '';
        if ( $directory ) {
            if ( is_numeric( $directory ) ) {
                return absint( $directory );
            }

            $taxonomy = defined( 'ATBDP_DIRECTORY_TYPE' ) ? ATBDP_DIRECTORY_TYPE : ( defined( 'ATBDP_TYPE' ) ? ATBDP_TYPE : 'atbdp_listing_types' );
            $term = get_term_by( 'slug', $directory, $taxonomy );
            if ( $term ) {
                return absint( $term->term_id );
            }
        }

        return 0;
    }

    public function register_rest_upload_fix() {
        add_filter( 'rest_request_before_callbacks', array( $this, 'rest_temp_upload_fix' ), 10, 3 );
        add_action( 'rest_api_init', array( $this, 'register_temp_media_upload_fallback' ) );
    }

    public function rest_temp_upload_fix( $response, $handler, $request ) {
        if ( ! $request instanceof WP_REST_Request ) {
            return $response;
        }

        $route = $request->get_route();
        if ( strpos( $route, '/directorist/v1/temp-media-upload' ) === false ) {
            return $response;
        }

        if ( strtoupper( $request->get_method() ) !== 'POST' ) {
            return $response;
        }

        $directory_id = $this->resolve_directory_id_from_request( $request );
        if ( ! $directory_id ) {
            return $response;
        }

        $logo_field_key = $this->ensure_logo_field_key_from_directory( $directory_id );
        if ( $logo_field_key ) {
            $field_param = $request->get_param( 'field' );
            if ( $field_param && $field_param !== $logo_field_key ) {
                $request->set_param( 'field', $logo_field_key );
            }
        }

        return $response;
    }

    public function register_temp_media_upload_fallback() {
        if ( ! function_exists( 'register_rest_route' ) ) {
            return;
        }

        $route = '/directorist/v1/temp-media-upload';
        $server = rest_get_server();
        $routes = $server ? $server->get_routes() : array();

        if ( isset( $routes[ $route ] ) ) {
            return;
        }

        register_rest_route(
            'directorist/v1',
            '/temp-media-upload',
            array(
                array(
                    'methods'             => WP_REST_Server::CREATABLE,
                    'callback'            => array( $this, 'handle_temp_media_upload_fallback' ),
                    'permission_callback' => array( $this, 'temp_media_upload_permissions_check' ),
                ),
            )
        );
    }

    public function temp_media_upload_permissions_check( WP_REST_Request $request ) {
        $nonce = $request->get_header( 'X-WP-Nonce' );

        if ( $nonce && wp_verify_nonce( $nonce, 'wp_rest' ) ) {
            return true;
        }

        if ( current_user_can( 'upload_files' ) ) {
            return true;
        }

        $allow_guest = function_exists( 'get_directorist_option' ) ? get_directorist_option( 'guest_listings', false ) : false;
        if ( $allow_guest ) {
            return true;
        }

        return new WP_Error(
            'directorist_rest_cannot_create',
            __( 'Sorry, you are not allowed to upload any file.', 'directorist-logo-field' ),
            array( 'status' => 400 )
        );
    }

    public function handle_temp_media_upload_fallback( WP_REST_Request $request ) {
        $directory = $request->get_param( 'directory' );
        $field_key = $request->get_param( 'field' );

        if ( empty( $directory ) ) {
            return new WP_Error( 'invalid_directory', __( 'Invalid directory.', 'directorist-logo-field' ), array( 'status' => 400 ) );
        }

        if ( empty( $field_key ) ) {
            return new WP_Error( 'invalid_field', __( 'Invalid field.', 'directorist-logo-field' ), array( 'status' => 400 ) );
        }

        $directory_id = $this->resolve_directory_id_from_request( $request );
        if ( empty( $directory_id ) ) {
            return new WP_Error( 'invalid_directory', __( 'Invalid directory.', 'directorist-logo-field' ), array( 'status' => 400 ) );
        }

        $this->ensure_logo_field_key_from_directory( $directory_id );
        $field = wp_list_filter( directorist_get_listing_form_fields( (int) $directory_id ), array( 'field_key' => $field_key ) );
        if ( empty( $field ) ) {
            return new WP_Error( 'invalid_field', __( 'Invalid field.', 'directorist-logo-field' ), array( 'status' => 400 ) );
        }

        $field = current( $field );
        $files = $request->get_file_params();

        if ( empty( $files ) || empty( $files['file'] ) ) {
            return new WP_Error( 'directorist_rest_upload_no_data', __( 'No data supplied.', 'directorist-logo-field' ), array( 'status' => 400 ) );
        }

        require_once ABSPATH . 'wp-admin/includes/file.php';

        add_filter( 'upload_dir', array( __CLASS__, 'set_temporary_upload_dir' ) );

        $maybe_mime_group = $field['file_type'] ?? 'image';
        $status = wp_handle_upload(
            $files['file'],
            array(
                'test_form' => false,
                'test_type' => true,
                'mimes'     => $this->get_supported_mime_types( $maybe_mime_group ),
            )
        );

        remove_filter( 'upload_dir', array( __CLASS__, 'set_temporary_upload_dir' ) );

        if ( ! empty( $status['error'] ) ) {
            return new WP_Error( 'directorist_rest_upload_error', $status['error'], array( 'status' => 500 ) );
        }

        if ( empty( $status['url'] ) ) {
            return new WP_Error( 'directorist_rest_upload_error', __( 'Could not upload file.', 'directorist-logo-field' ), array( 'status' => 500 ) );
        }

        $data = array(
            'name' => $files['file']['name'],
            'file' => basename( $status['url'] ),
        );

        return rest_ensure_response( $data );
    }

    public static function set_temporary_upload_dir( $upload ) {
        $upload['subdir'] = DIRECTORY_SEPARATOR . directorist_get_temp_upload_dir() . DIRECTORY_SEPARATOR . date( 'nj' );
        $upload['path']   = $upload['basedir'] . $upload['subdir'];
        $upload['url']    = $upload['baseurl'] . $upload['subdir'];

        return $upload;
    }

    private function get_supported_mime_types( $file_type = 'image' ) {
        if ( in_array( $file_type, array( '', 'all_types', 'all' ), true ) ) {
            $file_types = directorist_get_supported_file_types();
        } else {
            $groups = directorist_get_supported_file_types_groups();
            if ( isset( $groups[ $file_type ] ) ) {
                $file_types = $groups[ $file_type ];
            } else {
                $file_types = (array) $file_type;
            }
        }

        $supported_mimes = array();
        foreach ( get_allowed_mime_types() as $ext => $mime ) {
            $_exts = explode( '|', $ext );
            $match = array_intersect( $file_types, $_exts );
            if ( count( $match ) ) {
                $supported_mimes[ $ext ] = $mime;
            }
        }

        return $supported_mimes;
    }

    private function resolve_directory_id_from_request( WP_REST_Request $request ) {
        $directory = $request->get_param( 'directory' );
        if ( empty( $directory ) ) {
            return 0;
        }

        if ( is_numeric( $directory ) ) {
            return absint( $directory );
        }

        $taxonomy = defined( 'ATBDP_DIRECTORY_TYPE' ) ? ATBDP_DIRECTORY_TYPE : ( defined( 'ATBDP_TYPE' ) ? ATBDP_TYPE : 'atbdp_listing_types' );
        $term = get_term_by( 'slug', sanitize_text_field( $directory ), $taxonomy );
        if ( $term ) {
            return absint( $term->term_id );
        }

        return 0;
    }

    private function ensure_logo_field_key_from_directory( $directory_id ) {
        $form_data = get_term_meta( $directory_id, 'submission_form_fields', true );
        if ( empty( $form_data ) || empty( $form_data['fields'] ) || ! is_array( $form_data['fields'] ) ) {
            return '';
        }

        $logo_field_key = '';
        $changed = false;

        foreach ( $form_data['fields'] as $key => $field ) {
            if ( ! is_array( $field ) ) {
                continue;
            }

            $field_type = isset( $field['widget_name'] ) ? $field['widget_name'] : ( $field['type'] ?? '' );
            if ( 'logo' !== $field_type ) {
                continue;
            }

            if ( empty( $field['field_key'] ) ) {
                $field['field_key'] = 'directorist_logo';
                $form_data['fields'][ $key ] = $field;
                $changed = true;
            }

            $logo_field_key = $field['field_key'];
            break;
        }

        if ( $changed ) {
            update_term_meta( $directory_id, 'submission_form_fields', $form_data );
        }

        return $logo_field_key;
    }

    /**
     * Register field template path
     */
    public function register_field_template( $template, $field_data ) {
        $field_type = '';

        if ( is_array( $field_data ) ) {
            $field_type = isset( $field_data['widget_name'] ) ? $field_data['widget_name'] : '';
            if ( ! $field_type ) {
                $field_type = isset( $field_data['type'] ) ? $field_data['type'] : '';
            }
        } else {
            $field_type = $field_data;
        }

        if ( 'logo' === $field_type ) {
            return 'listing-form/custom-fields/logo';
        }

        return $template;
    }

    /**
     * Get field template
     */
    public function get_field_template( $template, $field_data ) {
        return DIRECTORIST_LOGO_FIELD_PATH . 'templates/form-field-logo.php';
    }

    /**
     * Provide the actual file path for our custom template
     */
    public function filter_template_file_path( $path, $template_name, $args ) {
        if ( 'listing-form/custom-fields/logo' === $template_name || 'listing-form/fields/logo' === $template_name ) {
            $custom = DIRECTORIST_LOGO_FIELD_PATH . 'templates/form-field-logo.php';
            if ( file_exists( $custom ) ) {
                return $custom;
            }
        }

        if ( 'logo-display' === $template_name ) {
            $custom = DIRECTORIST_LOGO_FIELD_PATH . 'templates/logo-display.php';
            if ( file_exists( $custom ) ) {
                return $custom;
            }
        }

        return $path;
    }

    /**
     * Filter field HTML output
     */
    public function filter_field_html( $html, $field_data, $listing_id ) {
        if ( ! isset( $field_data['widget_name'] ) && ! isset( $field_data['type'] ) ) {
            return $html;
        }
        
        $field_type = isset( $field_data['widget_name'] ) ? $field_data['widget_name'] : $field_data['type'];
        
        if ( 'logo' !== $field_type ) {
            return $html;
        }
        
        ob_start();
        $this->render_form_field( $field_data, $listing_id );
        return ob_get_clean();
    }

    /**
     * Render form field for add/edit listing
     */
    public function render_form_field( $field_data, $listing_id = 0 ) {
        if ( ! $listing_id && isset( $_GET['edit'] ) ) {
            $listing_id = absint( $_GET['edit'] );
        }

        $this->ensure_logo_field_key();

        $field_data['listing_id'] = $listing_id;
        $data = $field_data;
        include DIRECTORIST_LOGO_FIELD_PATH . 'templates/form-field-logo.php';
    }

    /**
     * Render the upload field HTML
     */
    private function render_upload_field( $field_key, $label, $placeholder, $required, $max_size, $logo_id, $logo_url ) {
        $has_logo = ! empty( $logo_url );
        $required_attr = $required ? 'required' : '';
        $required_class = $required ? 'directorist-form-required' : '';
        ?>
        <div class="directorist-form-group directorist-logo-field-wrap <?php echo esc_attr( $required_class ); ?>">
            <label class="directorist-form-label">
                <?php echo esc_html( $label ); ?>
                <?php if ( $required ) : ?>
                    <span class="directorist-form-required-mark">*</span>
                <?php endif; ?>
            </label>
            
            <div class="directorist-logo-upload-area <?php echo $has_logo ? 'has-logo' : ''; ?>" 
                 data-max-size="<?php echo esc_attr( $max_size ); ?>">
                
                <input type="hidden" 
                       name="directorist_logo" 
                       id="directorist_logo_id" 
                       value="<?php echo esc_attr( $logo_id ); ?>"
                       class="directorist-logo-input" />
                <?php if ( ! empty( $field_key ) && 'directorist_logo' !== $field_key ) : ?>
                    <input type="hidden"
                           name="<?php echo esc_attr( $field_key ); ?>"
                           id="<?php echo esc_attr( $field_key ); ?>_logo_id"
                           value="<?php echo esc_attr( $logo_id ); ?>"
                           class="directorist-logo-input directorist-logo-input--alias" />
                <?php endif; ?>
                
                <div class="directorist-logo-preview" <?php echo ! $has_logo ? 'style="display:none;"' : ''; ?>>
                    <img src="<?php echo esc_url( $logo_url ); ?>" alt="<?php echo esc_attr( $label ); ?>" />
                    <button type="button" class="directorist-logo-remove" title="<?php esc_attr_e( 'Remove logo', 'directorist-logo-field' ); ?>">
                        <span class="la la-times"></span>
                    </button>
                </div>
                
                <div class="directorist-logo-upload-btn" <?php echo $has_logo ? 'style="display:none;"' : ''; ?>>
                    <span class="la la-cloud-upload"></span>
                    <span class="directorist-logo-placeholder"><?php echo esc_html( $placeholder ); ?></span>
                    <span class="directorist-logo-hint">
                        <?php printf( 
                            esc_html__( 'JPG, PNG, GIF or WebP. Max %s KB', 'directorist-logo-field' ),
                            number_format( $max_size )
                        ); ?>
                    </span>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Save logo field value on listing insert/update
     */
    public function save_logo_field( $listing_id, $data = array(), $directory_type = '' ) {
        $this->save_logo_meta( $listing_id );
    }

    /**
     * Save on form submit
     */
    public function save_on_submit( $listing_id, $data ) {
        $this->save_logo_meta( $listing_id );
    }

    /**
     * Save on post save (admin)
     */
    public function save_on_post_save( $post_id, $post, $update ) {
        // Verify this is not an autosave
        if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
            return;
        }
        
        // Check post type
        if ( 'at_biz_dir' !== $post->post_type ) {
            return;
        }
        
        $this->save_logo_meta( $post_id );
    }

    /**
     * Save logo meta data
     */
    private function save_logo_meta( $listing_id ) {
        // Check for logo in POST data
        if ( isset( $_POST['directorist_logo'] ) && is_array( $_POST['directorist_logo'] ) ) {
            return;
        }

        if ( isset( $_POST['logo'] ) && is_array( $_POST['logo'] ) ) {
            return;
        }

        $logo_id = 0;
        if ( isset( $_POST['directorist_logo'] ) ) {
            $logo_id = absint( $_POST['directorist_logo'] );
        } elseif ( isset( $_POST['logo'] ) ) {
            $logo_id = absint( $_POST['logo'] );
        }

        if ( $logo_id || isset( $_POST['directorist_logo'] ) || isset( $_POST['logo'] ) ) {
            if ( $logo_id ) {
                // Verify attachment exists and is an image
                if ( wp_attachment_is_image( $logo_id ) ) {
                    update_post_meta( $listing_id, '_directorist_logo', $logo_id );
                }
            } else {
                delete_post_meta( $listing_id, '_directorist_logo' );
            }
        }
    }

    /**
     * Handle logo temp uploads on frontend submission
     */
    public function handle_logo_uploads( $listing_id ) {
        if ( empty( $listing_id ) || empty( $_POST ) ) {
            return;
        }

        $field_key = $this->get_logo_field_key();
        if ( ! isset( $_POST[ $field_key ] ) && ! isset( $_POST[ $field_key . '_old' ] ) ) {
            if ( isset( $_POST['directorist_logo'] ) || isset( $_POST['directorist_logo_old'] ) ) {
                $field_key = 'directorist_logo';
            } elseif ( isset( $_POST['logo'] ) || isset( $_POST['logo_old'] ) ) {
                $field_key = 'logo';
            }
        }

        $has_new = isset( $_POST[ $field_key ] );
        $has_old = isset( $_POST[ $field_key . '_old' ] );

        if ( ! $has_new && ! $has_old ) {
            return;
        }

        $previous_logo_id = absint( get_post_meta( $listing_id, '_directorist_logo', true ) );

        $new_images = array();
        $old_images = array();

        if ( $has_new ) {
            $new_images = (array) wp_unslash( $_POST[ $field_key ] );
        }

        if ( $has_old ) {
            $old_images = (array) wp_unslash( $_POST[ $field_key . '_old' ] );
        }

        $new_images = array_filter( $new_images );
        $old_images = array_filter( wp_parse_id_list( $old_images ) );

        $maybe_old = array_filter( $new_images, 'is_numeric' );
        if ( ! empty( $maybe_old ) ) {
            $old_images = array_merge( $old_images, wp_parse_id_list( $maybe_old ) );
            $new_images = array_diff( $new_images, $maybe_old );
        }

        if ( empty( $new_images ) && empty( $old_images ) ) {
            if ( $previous_logo_id ) {
                wp_delete_attachment( $previous_logo_id, true );
            }
            delete_post_meta( $listing_id, '_directorist_logo' );
            return;
        }

        $new_logo_id = 0;
        if ( ! empty( $new_images ) ) {
            $new_logo_id = $this->import_temp_logo( $listing_id, $new_images[0] );
        }

        $final_logo_id = $new_logo_id ? $new_logo_id : ( ! empty( $old_images ) ? absint( $old_images[0] ) : 0 );

        if ( $final_logo_id ) {
            update_post_meta( $listing_id, '_directorist_logo', $final_logo_id );
        } else {
            delete_post_meta( $listing_id, '_directorist_logo' );
        }

        if ( $previous_logo_id && $previous_logo_id !== $final_logo_id ) {
            wp_delete_attachment( $previous_logo_id, true );
        }

        if ( ! empty( $old_images ) ) {
            foreach ( array_slice( $old_images, 1 ) as $extra_old ) {
                if ( $extra_old && $extra_old !== $final_logo_id ) {
                    wp_delete_attachment( $extra_old, true );
                }
            }
        }
    }

    private function import_temp_logo( $listing_id, $filename ) {
        $filename = sanitize_file_name( $filename );
        if ( empty( $filename ) ) {
            return 0;
        }

        $upload_dir = wp_get_upload_dir();
        $temp_dir   = trailingslashit( $upload_dir['basedir'] ) . trailingslashit( directorist_get_temp_upload_dir() . DIRECTORY_SEPARATOR . date( 'nj' ) );
        $target_dir = trailingslashit( $upload_dir['path'] );
        $filepath   = $temp_dir . $filename;

        if ( is_dir( $filepath ) || ! file_exists( $filepath ) ) {
            return 0;
        }

        if ( file_exists( $target_dir . $filename ) ) {
            $filename = wp_unique_filename( $target_dir, $filename );
        }

        rename( $filepath, $target_dir . $filename );

        $mime = wp_check_filetype( $filename );
        $name = wp_basename( $filename, ".{$mime['ext']}" );

        $attachment = array(
            'post_mime_type' => $mime['type'],
            'guid'           => trailingslashit( $upload_dir['url'] ) . $filename,
            'post_parent'    => $listing_id,
            'post_title'     => sanitize_text_field( $name ),
        );

        $attachment_id = $this->run_with_upload_capability(
            function() use ( $attachment, $target_dir, $filename, $listing_id ) {
                $attachment_id = wp_insert_attachment( $attachment, $target_dir . $filename, $listing_id, false );
                if ( is_wp_error( $attachment_id ) ) {
                    return $attachment_id;
                }

                if ( function_exists( 'directorist_background_image_process' ) ) {
                    directorist_background_image_process( array( $attachment_id => $target_dir . $filename ) );
                } else {
                    require_once ABSPATH . 'wp-admin/includes/image.php';
                    $meta = wp_generate_attachment_metadata( $attachment_id, $target_dir . $filename );
                    if ( $meta && ! is_wp_error( $meta ) ) {
                        wp_update_attachment_metadata( $attachment_id, $meta );
                    }
                }

                return $attachment_id;
            }
        );

        if ( is_wp_error( $attachment_id ) ) {
            return 0;
        }

        return $attachment_id;
    }

    /**
     * AJAX handler for logo upload
     */
    public function ajax_upload_logo() {
        check_ajax_referer( 'directorist_logo_field_nonce', 'nonce' );

        // Check permissions - allow logged in users or guests if enabled
        if ( ! is_user_logged_in() ) {
            $allow_guest = function_exists( 'get_directorist_option' ) ? get_directorist_option( 'guest_listings', false ) : false;
            if ( ! $allow_guest ) {
                wp_send_json_error( array( 'message' => __( 'You must be logged in to upload files.', 'directorist-logo-field' ) ) );
            }
        }

        if ( empty( $_FILES['logo_file'] ) ) {
            wp_send_json_error( array( 'message' => __( 'No file uploaded.', 'directorist-logo-field' ) ) );
        }

        require_once( ABSPATH . 'wp-admin/includes/image.php' );
        require_once( ABSPATH . 'wp-admin/includes/file.php' );
        require_once( ABSPATH . 'wp-admin/includes/media.php' );

        // Validate file type
        $allowed_types = array( 'image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'image/webp' );
        
        if ( ! in_array( $_FILES['logo_file']['type'], $allowed_types ) ) {
            wp_send_json_error( array( 'message' => __( 'Invalid file type. Please upload a JPG, PNG, GIF, or WebP image.', 'directorist-logo-field' ) ) );
        }

        // Handle the upload
        $attachment_id = $this->run_with_upload_capability(
            function() {
                return media_handle_upload( 'logo_file', 0 );
            }
        );

        if ( is_wp_error( $attachment_id ) ) {
            wp_send_json_error( array( 'message' => $attachment_id->get_error_message() ) );
        }

        $attachment_url = wp_get_attachment_image_url( $attachment_id, 'medium' );

        wp_send_json_success( array(
            'id'  => $attachment_id,
            'url' => $attachment_url,
        ) );
    }

    private function run_with_upload_capability( callable $callback ) {
        $allow_guest = function_exists( 'get_directorist_option' ) ? get_directorist_option( 'guest_listings', false ) : false;

        if ( current_user_can( 'upload_files' ) || ( ! is_user_logged_in() && ! $allow_guest ) ) {
            return $callback();
        }

        $user_id = get_current_user_id();
        $cap_filter = function( $allcaps, $caps, $args, $user ) use ( $user_id, $allow_guest ) {
            if ( $user_id > 0 ) {
                if ( $user instanceof WP_User && (int) $user->ID !== (int) $user_id ) {
                    return $allcaps;
                }
            } elseif ( ! $allow_guest ) {
                return $allcaps;
            }

            $allcaps['upload_files'] = true;
            return $allcaps;
        };

        add_filter( 'user_has_cap', $cap_filter, 10, 4 );

        try {
            return $callback();
        } finally {
            remove_filter( 'user_has_cap', $cap_filter, 10 );
        }
    }
}
