<?php
/**
 * Logo Field Template Injection
 * 
 * Injects logo using JavaScript for reliable display
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class Directorist_Logo_Field_Inject {

    /**
     * Constructor
     */
    public function __construct() {
        // Output logo data and JavaScript injection
        add_action( 'wp_footer', array( $this, 'inject_logos_via_js' ), 100 );
        
        // Shortcodes for manual use
        add_shortcode( 'directorist_listings_with_logos', array( $this, 'listings_with_logos_shortcode' ) );
        add_shortcode( 'listing_logo', array( $this, 'single_logo_shortcode' ) );
        add_shortcode( 'directorist_logo', array( $this, 'single_logo_shortcode' ) );
    }

    /**
     * Inject logos via JavaScript
     */
    public function inject_logos_via_js() {
        $show_single = Directorist_Logo_Field_Settings::get_option( 'show_on_single', 1 );
        $show_listings = Directorist_Logo_Field_Settings::get_option( 'show_on_listings', 1 );
        
        // If both are disabled, don't output anything
        if ( ! $show_single && ! $show_listings ) {
            return;
        }
        
        $single_width = Directorist_Logo_Field_Settings::get_option( 'single_width', 180 );
        $listings_width = Directorist_Logo_Field_Settings::get_option( 'listings_width', 100 );
        
        // Get logo data for current page
        $logo_data = array();
        
        // For single listing page
        if ( is_singular( 'at_biz_dir' ) && $show_single ) {
            global $post;
            $logo_id = get_post_meta( $post->ID, '_directorist_logo', true );
            if ( $logo_id ) {
                $logo_url = wp_get_attachment_image_url( $logo_id, 'medium' );
                if ( $logo_url ) {
                    $logo_data['single'] = array(
                        'id' => $post->ID,
                        'url' => $logo_url,
                        'alt' => sprintf( __( '%s logo', 'directorist-logo-field' ), get_the_title( $post->ID ) ),
                        'width' => $single_width,
                    );
                }
            }
        }
        
        // For archive/listings pages - get all listing logos
        if ( $show_listings ) {
            // Query all listings that have logos
            $listings_with_logos = get_posts( array(
                'post_type' => 'at_biz_dir',
                'posts_per_page' => 100,
                'post_status' => 'publish',
                'meta_query' => array(
                    array(
                        'key' => '_directorist_logo',
                        'compare' => 'EXISTS',
                    ),
                ),
                'fields' => 'ids',
            ) );
            
            if ( ! empty( $listings_with_logos ) ) {
                $logo_data['listings'] = array();
                foreach ( $listings_with_logos as $listing_id ) {
                    $logo_id = get_post_meta( $listing_id, '_directorist_logo', true );
                    if ( $logo_id ) {
                        $logo_url = wp_get_attachment_image_url( $logo_id, 'medium' );
                        if ( $logo_url ) {
                            $logo_data['listings'][ $listing_id ] = array(
                                'id' => $listing_id,
                                'url' => $logo_url,
                                'alt' => sprintf( __( '%s logo', 'directorist-logo-field' ), get_the_title( $listing_id ) ),
                                'permalink' => get_permalink( $listing_id ),
                            );
                        }
                    }
                }
                $logo_data['listings_width'] = $listings_width;
            }
        }
        
        // If no logos to display, don't output script
        if ( empty( $logo_data ) ) {
            return;
        }
        
        ?>
        <script type="text/javascript">
        (function() {
            var logoData = <?php echo json_encode( $logo_data ); ?>;
            
            function injectLogos() {
                // Single listing page logo
                if (logoData.single) {
                    var singleLogo = logoData.single;
                    
                    // Create logo HTML
                    var logoHtml = '<div class="directorist-logo-injected directorist-single-logo-wrapper" style="display: block; margin: 0 0 20px 0; padding: 0;">' +
                        '<img src="' + singleLogo.url + '" alt="' + singleLogo.alt + '" ' +
                        'class="directorist-logo-image directorist-single-logo" ' +
                        'style="width: ' + singleLogo.width + 'px; max-width: 100%; height: auto; display: block;" loading="lazy" />' +
                        '</div>';
                    
                    // Try to find the best insertion point - after the slider/header
                    var insertPoints = [
                        '.directorist-single-contents-section',
                        '.directorist-single-listing-section',
                        '.directorist-single-details-section',
                        '.directorist-single-listing__details',
                        '.directorist-single-listing-details',
                        '.atbd_content_module',
                        '.atbdp-single-listing-wrapper .container > div:nth-child(2)',
                        '.atbdp-single-listing-wrapper .row > div:first-child',
                        '.directorist-single-wrapper',
                        '#directorist-single-listing-content',
                        '.atbd_listing_details',
                        'article.at_biz_dir .entry-content',
                        '.single-at_biz_dir .entry-content',
                        '.single-at_biz_dir article'
                    ];
                    
                    var inserted = false;
                    for (var i = 0; i < insertPoints.length; i++) {
                        var target = document.querySelector(insertPoints[i]);
                        if (target && !document.querySelector('.directorist-single-logo-wrapper')) {
                            // Insert at the beginning of this element
                            target.insertAdjacentHTML('afterbegin', logoHtml);
                            inserted = true;
                            console.log('Logo inserted at: ' + insertPoints[i]);
                            break;
                        }
                    }
                    
                    // Fallback: insert after the slider if found
                    if (!inserted) {
                        var slider = document.querySelector('.directorist-single-slider, .swiper-container, .directorist-listing-slider, .atbdp-single-slider');
                        if (slider && slider.parentNode && !document.querySelector('.directorist-single-logo-wrapper')) {
                            slider.insertAdjacentHTML('afterend', logoHtml);
                            inserted = true;
                            console.log('Logo inserted after slider');
                        }
                    }
                    
                    if (!inserted) {
                        console.log('Could not find insertion point for single logo');
                    }
                }
                
                // Listing cards logos
                if (logoData.listings && Object.keys(logoData.listings).length > 0) {
                    var width = logoData.listings_width || 100;
                    var listingsByUrl = {};
                    
                    Object.keys(logoData.listings).forEach(function(listingId) {
                        var listing = logoData.listings[listingId];
                        if (listing && listing.permalink) {
                            listingsByUrl[normalizeUrl(listing.permalink)] = listing;
                        }
                    });
                    
                    function buildCardLogoHtml(logo) {
                        var href = logo.permalink || '#';
                        return '<div class="directorist-logo-injected directorist-card-logo-wrapper" style="display: block; margin: 0; padding: 0; text-align: left;">' +
                            '<a class="directorist-logo-link" href="' + href + '">' +
                            '<img src="' + logo.url + '" alt="' + logo.alt + '" ' +
                            'class="directorist-logo-image directorist-card-logo" ' +
                            'style="width: ' + width + 'px; max-width: 100%; height: auto; display: block;" loading="lazy" />' +
                            '</a>' +
                            '</div>';
                    }
                    
                    function insertLogoAboveTitle(card, logo) {
                        if (!card || !logo || card.querySelector('.directorist-card-logo-wrapper')) {
                            return;
                        }
                        
                        var titleEl = card.querySelector('.directorist-listing-card__title, .directorist-listing-title, .atbd_listing_title, h1, h2, h3, h4, h5');
                        if (titleEl) {
                            var titleContainer = titleEl.closest('.directorist-listing-card__title, .directorist-listing-title, .atbd_listing_title, h1, h2, h3, h4, h5') || titleEl;
                            titleContainer.insertAdjacentHTML('beforebegin', buildCardLogoHtml(logo));
                            return;
                        }
                        
                        var bodyEl = card.querySelector('.directorist-listing-card__body, .directorist-listing-body, .atbd_listing_info, .card-body');
                        if (bodyEl) {
                            bodyEl.insertAdjacentHTML('afterbegin', buildCardLogoHtml(logo));
                            return;
                        }
                        
                        card.insertAdjacentHTML('afterbegin', buildCardLogoHtml(logo));
                    }
                    
                    function getListingDataFromCard(card) {
                        var listingId = extractListingIdFromCard(card);
                        if (listingId && logoData.listings[listingId]) {
                            return logoData.listings[listingId];
                        }
                        
                        var link = card.querySelector('.directorist-listing-card__title a, .directorist-listing-title a, .atbd_listing_title a, a[href]');
                        if (link) {
                            var normalizedLink = normalizeUrl(link.getAttribute('href'));
                            if (normalizedLink && listingsByUrl[normalizedLink]) {
                                return listingsByUrl[normalizedLink];
                            }
                        }
                        
                        return null;
                    }
                    
                    // Find all listing cards
                    var cardSelectors = [
                        '.directorist-listing-single',
                        '.directorist-listing-card',
                        '.directorist-listing-card__single',
                        '.atbdp-listing-card',
                        '.atbd_single_listing',
                        '.directorist-archive-card',
                        '[data-listing-id]'
                    ];
                    
                    cardSelectors.forEach(function(selector) {
                        var cards = document.querySelectorAll(selector);
                        cards.forEach(function(card) {
                            var logo = getListingDataFromCard(card);
                            if (logo) {
                                insertLogoAboveTitle(card, logo);
                            }
                        });
                    });
                    
                    // Also try finding cards by looking for links to listings
                    Object.keys(logoData.listings).forEach(function(listingId) {
                        var logo = logoData.listings[listingId];
                        if (!logo || !logo.permalink) {
                            return;
                        }
                        
                        var normalizedListingUrl = normalizeUrl(logo.permalink);
                        if (!normalizedListingUrl) {
                            return;
                        }
                        
                        var links = document.querySelectorAll('a[href]');
                        links.forEach(function(link) {
                            if (normalizeUrl(link.getAttribute('href')) !== normalizedListingUrl) {
                                return;
                            }
                            
                            var card = link.closest('.directorist-listing-single, .directorist-listing-card, .directorist-listing-card__single, .atbdp-listing-card, .atbd_single_listing, article');
                            if (card) {
                                insertLogoAboveTitle(card, logo);
                            }
                        });
                    });
                }
            }
            
            function extractListingIdFromCard(card) {
                var explicitId = card.getAttribute('data-listing-id') ||
                    card.getAttribute('data-id') ||
                    card.getAttribute('data-post-id');
                if (explicitId) {
                    return explicitId;
                }
                
                var dataIdEl = card.querySelector('[data-listing-id], [data-id], [data-post-id]');
                if (dataIdEl) {
                    return dataIdEl.getAttribute('data-listing-id') ||
                        dataIdEl.getAttribute('data-id') ||
                        dataIdEl.getAttribute('data-post-id');
                }
                
                // Try to extract ID from link
                var link = card.querySelector('a[href*="at_biz_dir"], a[href*="listing"], a[href*="?p="]');
                if (link) {
                    var href = link.getAttribute('href') || '';
                    var match = href.match(/[?&]p=(\d+)/) || href.match(/listing[\/\-](\d+)/) || href.match(/at_biz_dir[\/\-](\d+)/);
                    if (match) {
                        return match[1];
                    }
                }
                
                // Try to get from card classes or ID
                var cardId = card.id || '';
                var cardIdMatch = cardId.match(/(\d+)/);
                if (cardIdMatch) {
                    return cardIdMatch[1];
                }
                
                return null;
            }
            
            function normalizeUrl(url) {
                if (!url) {
                    return '';
                }
                
                var link = document.createElement('a');
                link.href = url;
                var normalized = link.href.split('#')[0].split('?')[0];
                normalized = normalized.replace(/\/$/, '');
                return normalized;
            }
            
            // Run when DOM is ready
            if (document.readyState === 'loading') {
                document.addEventListener('DOMContentLoaded', injectLogos);
            } else {
                injectLogos();
            }
            
            // Also run after a short delay to catch dynamically loaded content
            setTimeout(injectLogos, 1000);
            setTimeout(injectLogos, 2500);
            
            // Watch for dynamically injected cards (filters/pagination)
            if (window.MutationObserver) {
                var injectTimer;
                var observer = new MutationObserver(function() {
                    clearTimeout(injectTimer);
                    injectTimer = setTimeout(injectLogos, 150);
                });
                
                observer.observe(document.body, { childList: true, subtree: true });
            }
        })();
        </script>
        <?php
    }

    /**
     * Single logo shortcode
     */
    public function single_logo_shortcode( $atts ) {
        $atts = shortcode_atts( array(
            'width' => 180,
            'id' => 0,
        ), $atts );
        
        $listing_id = intval( $atts['id'] );
        if ( ! $listing_id ) {
            global $post;
            $listing_id = $post ? $post->ID : 0;
        }
        
        if ( ! $listing_id ) {
            return '';
        }
        
        $logo_id = get_post_meta( $listing_id, '_directorist_logo', true );
        if ( ! $logo_id ) {
            return '';
        }
        
        $logo_url = wp_get_attachment_image_url( $logo_id, 'medium' );
        if ( ! $logo_url ) {
            return '';
        }
        
        $alt = sprintf( __( '%s logo', 'directorist-logo-field' ), get_the_title( $listing_id ) );
        $width = intval( $atts['width'] );
        
        return '<div class="directorist-logo-shortcode" style="margin: 0 0 15px 0;">' .
               '<img src="' . esc_url( $logo_url ) . '" alt="' . esc_attr( $alt ) . '" ' .
               'style="width: ' . $width . 'px; max-width: 100%; height: auto;" loading="lazy" />' .
               '</div>';
    }

    /**
     * Listings with logos shortcode
     */
    public function listings_with_logos_shortcode( $atts ) {
        $atts = shortcode_atts( array(
            'number' => 10,
            'columns' => 3,
            'logo_width' => 120,
        ), $atts );
        
        ob_start();
        
        $args = array(
            'post_type' => 'at_biz_dir',
            'posts_per_page' => intval( $atts['number'] ),
            'post_status' => 'publish',
            'meta_query' => array(
                array(
                    'key' => '_directorist_logo',
                    'compare' => 'EXISTS',
                ),
            ),
        );
        
        $query = new WP_Query( $args );
        
        if ( $query->have_posts() ) {
            $columns = intval( $atts['columns'] );
            echo '<div class="directorist-logo-listings" style="display: grid; grid-template-columns: repeat(' . $columns . ', 1fr); gap: 20px; margin: 20px 0;">';
            
            while ( $query->have_posts() ) {
                $query->the_post();
                $logo_id = get_post_meta( get_the_ID(), '_directorist_logo', true );
                $logo_url = wp_get_attachment_image_url( $logo_id, 'medium' );
                ?>
                <div class="directorist-logo-listing-card" style="text-align: center; padding: 15px; border: 1px solid #eee; border-radius: 5px;">
                    <?php if ( $logo_url ) : ?>
                    <a href="<?php the_permalink(); ?>">
                        <img src="<?php echo esc_url( $logo_url ); ?>" 
                             alt="<?php the_title_attribute(); ?>" 
                             style="width: <?php echo intval( $atts['logo_width'] ); ?>px; max-width: 100%; height: auto; margin-bottom: 10px;" />
                    </a>
                    <?php endif; ?>
                    <h4 style="margin: 10px 0 5px;"><a href="<?php the_permalink(); ?>"><?php the_title(); ?></a></h4>
                </div>
                <?php
            }
            
            echo '</div>';
            wp_reset_postdata();
        }
        
        return ob_get_clean();
    }
}
