<?php
/**
 * Logo Field Settings
 * 
 * Adds settings panel for logo display options
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class Directorist_Logo_Field_Settings {

    /**
     * Constructor
     */
    public function __construct() {
        add_action( 'admin_menu', array( $this, 'add_settings_menu' ), 99 );
        add_action( 'admin_init', array( $this, 'register_settings' ) );
        
        // Add settings link on plugins page
        add_filter( 'plugin_action_links_' . plugin_basename( DIRECTORIST_LOGO_FIELD_FILE ), array( $this, 'add_settings_link' ) );
    }

    /**
     * Add settings link to plugins page
     */
    public function add_settings_link( $links ) {
        $settings_link = '<a href="' . admin_url( 'admin.php?page=directorist-logo-settings' ) . '">' . __( 'Settings', 'directorist-logo-field' ) . '</a>';
        array_unshift( $links, $settings_link );
        return $links;
    }

    /**
     * Add settings menu - try multiple parent menus for compatibility
     */
    public function add_settings_menu() {
        // Try adding under Directory Listings post type menu
        add_submenu_page(
            'edit.php?post_type=at_biz_dir',
            __( 'Logo Settings', 'directorist-logo-field' ),
            __( 'Logo Settings', 'directorist-logo-field' ),
            'manage_options',
            'directorist-logo-settings',
            array( $this, 'render_settings_page' )
        );
        
        // Also add as a top-level menu for easy access
        add_menu_page(
            __( 'Logo Settings', 'directorist-logo-field' ),
            __( 'Logo Field', 'directorist-logo-field' ),
            'manage_options',
            'directorist-logo-settings-main',
            array( $this, 'render_settings_page' ),
            'dashicons-format-image',
            81
        );
    }

    /**
     * Register settings
     */
    public function register_settings() {
        register_setting( 'directorist_logo_field_settings', 'directorist_logo_field_options', array(
            'sanitize_callback' => array( $this, 'sanitize_options' ),
            'default' => $this->get_defaults(),
        ) );

        // Single Listing Settings
        add_settings_section(
            'directorist_logo_single_section',
            __( 'Single Listing Page', 'directorist-logo-field' ),
            array( $this, 'single_section_description' ),
            'directorist-logo-settings'
        );

        add_settings_field(
            'show_on_single',
            __( 'Show Logo on Single Listing', 'directorist-logo-field' ),
            array( $this, 'render_checkbox_field' ),
            'directorist-logo-settings',
            'directorist_logo_single_section',
            array(
                'field' => 'show_on_single',
                'label' => __( 'Automatically display logo on single listing pages', 'directorist-logo-field' ),
            )
        );

        add_settings_field(
            'single_width',
            __( 'Logo Width (Single)', 'directorist-logo-field' ),
            array( $this, 'render_number_field' ),
            'directorist-logo-settings',
            'directorist_logo_single_section',
            array(
                'field' => 'single_width',
                'description' => __( 'Width in pixels. Height adjusts automatically.', 'directorist-logo-field' ),
            )
        );

        // All Listings Settings
        add_settings_section(
            'directorist_logo_listings_section',
            __( 'All Listings Page (Directory)', 'directorist-logo-field' ),
            array( $this, 'listings_section_description' ),
            'directorist-logo-settings'
        );

        add_settings_field(
            'show_on_listings',
            __( 'Show Logo on Listing Cards', 'directorist-logo-field' ),
            array( $this, 'render_checkbox_field' ),
            'directorist-logo-settings',
            'directorist_logo_listings_section',
            array(
                'field' => 'show_on_listings',
                'label' => __( 'Automatically display logo on each listing card', 'directorist-logo-field' ),
            )
        );

        add_settings_field(
            'listings_width',
            __( 'Logo Width (Listings)', 'directorist-logo-field' ),
            array( $this, 'render_number_field' ),
            'directorist-logo-settings',
            'directorist_logo_listings_section',
            array(
                'field' => 'listings_width',
                'description' => __( 'Width in pixels for listing cards.', 'directorist-logo-field' ),
            )
        );
    }

    /**
     * Get default options
     */
    public function get_defaults() {
        return array(
            'show_on_single' => 1,
            'single_width' => 180,
            'show_on_listings' => 1,
            'listings_width' => 100,
        );
    }

    /**
     * Get option value
     */
    public static function get_option( $key, $default = null ) {
        $options = get_option( 'directorist_logo_field_options', array() );
        $defaults = array(
            'show_on_single' => 1,
            'single_width' => 180,
            'show_on_listings' => 1,
            'listings_width' => 100,
        );
        
        $options = wp_parse_args( $options, $defaults );
        
        return isset( $options[ $key ] ) ? $options[ $key ] : $default;
    }

    /**
     * Sanitize options
     */
    public function sanitize_options( $input ) {
        $sanitized = array();
        
        $sanitized['show_on_single'] = ! empty( $input['show_on_single'] ) ? 1 : 0;
        $sanitized['single_width'] = absint( $input['single_width'] ?? 180 );
        
        $sanitized['show_on_listings'] = ! empty( $input['show_on_listings'] ) ? 1 : 0;
        $sanitized['listings_width'] = absint( $input['listings_width'] ?? 100 );
        
        return $sanitized;
    }

    /**
     * Section descriptions
     */
    public function single_section_description() {
        echo '<p>' . esc_html__( 'The logo will be displayed at the beginning of the listing content, outside of any image sliders.', 'directorist-logo-field' ) . '</p>';
    }

    public function listings_section_description() {
        echo '<p>' . esc_html__( 'The logo will be displayed on each listing card in the directory. This works automatically without needing to add it in the layout builder.', 'directorist-logo-field' ) . '</p>';
    }

    /**
     * Render checkbox field
     */
    public function render_checkbox_field( $args ) {
        $options = get_option( 'directorist_logo_field_options', $this->get_defaults() );
        $value = isset( $options[ $args['field'] ] ) ? $options[ $args['field'] ] : 0;
        ?>
        <label>
            <input type="checkbox" 
                   name="directorist_logo_field_options[<?php echo esc_attr( $args['field'] ); ?>]" 
                   value="1" 
                   <?php checked( $value, 1 ); ?> />
            <?php echo esc_html( $args['label'] ); ?>
        </label>
        <?php
    }

    /**
     * Render number field
     */
    public function render_number_field( $args ) {
        $options = get_option( 'directorist_logo_field_options', $this->get_defaults() );
        $value = isset( $options[ $args['field'] ] ) ? $options[ $args['field'] ] : 180;
        ?>
        <input type="number" 
               name="directorist_logo_field_options[<?php echo esc_attr( $args['field'] ); ?>]" 
               value="<?php echo esc_attr( $value ); ?>" 
               min="50" 
               max="500" 
               step="10" 
               style="width: 100px;" /> px
        <?php if ( ! empty( $args['description'] ) ) : ?>
            <p class="description"><?php echo esc_html( $args['description'] ); ?></p>
        <?php endif; ?>
        <?php
    }

    /**
     * Render settings page
     */
    public function render_settings_page() {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        
        // Check if settings were saved
        if ( isset( $_GET['settings-updated'] ) ) {
            add_settings_error( 'directorist_logo_messages', 'directorist_logo_message', __( 'Settings Saved', 'directorist-logo-field' ), 'updated' );
        }
        ?>
        <div class="wrap">
            <h1><?php esc_html_e( 'Directorist Logo Field Settings', 'directorist-logo-field' ); ?></h1>
            
            <?php settings_errors( 'directorist_logo_messages' ); ?>
            
            <form action="options.php" method="post">
                <?php
                settings_fields( 'directorist_logo_field_settings' );
                do_settings_sections( 'directorist-logo-settings' );
                submit_button( __( 'Save Settings', 'directorist-logo-field' ) );
                ?>
            </form>
            
            <hr />
            
            <h2><?php esc_html_e( 'How It Works', 'directorist-logo-field' ); ?></h2>
            <ol>
                <li><?php esc_html_e( 'Upload a logo for each listing using the "Listing Logo" box when editing a listing in WordPress admin.', 'directorist-logo-field' ); ?></li>
                <li><?php esc_html_e( 'Enable the options above to automatically display logos on your pages.', 'directorist-logo-field' ); ?></li>
                <li><?php esc_html_e( 'Logos will appear outside of any image sliders for clean display.', 'directorist-logo-field' ); ?></li>
            </ol>
            
            <h2><?php esc_html_e( 'Shortcode', 'directorist-logo-field' ); ?></h2>
            <p><?php esc_html_e( 'You can also display the logo manually using:', 'directorist-logo-field' ); ?></p>
            <code>[directorist_logo width="180"]</code>
        </div>
        <?php
    }
}
