<?php
/**
 * Logo Field Type Registration
 * 
 * Registers the custom logo field type with Directorist Form Builder and Layout Builders
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class Directorist_Logo_Field_Type {

    /**
     * Constructor
     */
    public function __construct() {
        $this->init_hooks();
    }

    /**
     * Initialize hooks
     */
    private function init_hooks() {
        // ========================================
        // FORM BUILDER - Add Listing Form
        // ========================================
        
        // Register field type in Form Builder custom widgets
        add_filter( 'atbdp_form_custom_widgets', array( $this, 'register_field_widget' ), 10, 1 );
        add_filter( 'directorist_form_custom_widgets', array( $this, 'register_field_widget' ), 10, 1 );
        
        // Add to submission form widgets
        add_filter( 'atbdp_submission_form_widgets', array( $this, 'add_submission_widget' ), 10, 1 );
        add_filter( 'directorist_submission_form_widgets', array( $this, 'add_submission_widget' ), 10, 1 );
        
        // Register as known widget type
        add_filter( 'directorist_form_fields', array( $this, 'register_form_field' ), 10, 1 );
        
        // Hook for rendering the widget in listing form
        add_action( 'atbdp_widget_logo', array( $this, 'render_form_widget' ), 10, 3 );
        add_action( 'directorist_listing_form_widget_logo', array( $this, 'render_form_widget' ), 10, 3 );
        
        // ========================================
        // SINGLE PAGE LAYOUT BUILDER - Section Widgets
        // ========================================
        
        // Listing header section widgets
        add_filter( 'atbdp_single_listing_header_widgets', array( $this, 'add_single_header_widget' ), 10, 1 );
        add_filter( 'directorist_single_listing_header_widgets', array( $this, 'add_single_header_widget' ), 10, 1 );
        
        // Listing content section widgets  
        add_filter( 'atbdp_single_listing_content_widgets', array( $this, 'add_single_content_widget' ), 10, 1 );
        add_filter( 'directorist_single_listing_content_widgets', array( $this, 'add_single_content_widget' ), 10, 1 );
        
        // Quick info section widgets
        add_filter( 'atbdp_single_listing_quick_info_widgets', array( $this, 'add_single_listing_widget' ), 10, 1 );
        
        // General available widgets  
        add_filter( 'atbdp_single_listing_available_widgets', array( $this, 'add_single_listing_widget' ), 10, 1 );
        add_filter( 'atbdp_single_listings_available_widgets', array( $this, 'add_single_listing_widget' ), 10, 1 );
        
        // Single listing fields filter
        add_filter( 'atbdp_single_listing_fields', array( $this, 'register_single_listing_widget' ), 10, 1 );
        
        // Single listing builder widgets - for all sections
        add_filter( 'directorist_single_listing_widgets', array( $this, 'add_builder_single_widget' ), 10, 1 );
        
        // ========================================
        // DIRECTORIST 7+ BUILDER SYSTEM
        // ========================================
        
        // Register in the builder's widget registry
        add_filter( 'directorist_single_listing_section_widgets', array( $this, 'add_section_widget' ), 10, 2 );
        add_filter( 'atbdp_single_listing_section_widgets', array( $this, 'add_section_widget' ), 10, 2 );
        
        // Widget group registration for different sections
        add_filter( 'directorist_single_listing_widget_groups', array( $this, 'add_to_widget_groups' ), 10, 1 );
        
        // ========================================
        // ALL LISTINGS / CARD LAYOUT BUILDER
        // ========================================
        
        // Main card widget registrations
        add_filter( 'atbdp_all_listings_card_widgets', array( $this, 'add_card_widget' ), 10, 1 );
        add_filter( 'atbdp_listings_card_widgets', array( $this, 'add_card_widget' ), 10, 1 );
        add_filter( 'directorist_all_listings_widgets', array( $this, 'add_card_widget' ), 10, 1 );
        add_filter( 'directorist_listing_card_widgets', array( $this, 'add_card_widget' ), 10, 1 );
        add_filter( 'directorist_all_listings_available_widgets', array( $this, 'add_card_widget' ), 10, 1 );
        
        // Grid View - specific area widgets (Avatar, Title, Info, Footer areas)
        add_filter( 'directorist_listing_card_grid_avatar_area_widgets', array( $this, 'add_card_widget' ), 10, 1 );
        add_filter( 'directorist_listing_card_grid_title_area_widgets', array( $this, 'add_card_widget' ), 10, 1 );
        add_filter( 'directorist_listing_card_grid_info_area_widgets', array( $this, 'add_card_widget' ), 10, 1 );
        add_filter( 'directorist_listing_card_grid_quick_info_widgets', array( $this, 'add_card_widget' ), 10, 1 );
        add_filter( 'directorist_listing_card_grid_footer_widgets', array( $this, 'add_card_widget' ), 10, 1 );
        
        // List View - specific area widgets
        add_filter( 'directorist_listing_card_list_avatar_area_widgets', array( $this, 'add_card_widget' ), 10, 1 );
        add_filter( 'directorist_listing_card_list_title_area_widgets', array( $this, 'add_card_widget' ), 10, 1 );
        add_filter( 'directorist_listing_card_list_info_area_widgets', array( $this, 'add_card_widget' ), 10, 1 );
        add_filter( 'directorist_listing_card_list_quick_info_widgets', array( $this, 'add_card_widget' ), 10, 1 );
        add_filter( 'directorist_listing_card_list_footer_widgets', array( $this, 'add_card_widget' ), 10, 1 );
        
        // ATBDP prefix variants
        add_filter( 'atbdp_listing_card_grid_avatar_area_widgets', array( $this, 'add_card_widget' ), 10, 1 );
        add_filter( 'atbdp_listing_card_grid_title_area_widgets', array( $this, 'add_card_widget' ), 10, 1 );
        add_filter( 'atbdp_listing_card_grid_info_area_widgets', array( $this, 'add_card_widget' ), 10, 1 );
        add_filter( 'atbdp_listing_card_grid_quick_info_widgets', array( $this, 'add_card_widget' ), 10, 1 );
        add_filter( 'atbdp_listing_card_list_avatar_area_widgets', array( $this, 'add_card_widget' ), 10, 1 );
        add_filter( 'atbdp_listing_card_list_title_area_widgets', array( $this, 'add_card_widget' ), 10, 1 );
        
        // Generic card section widgets
        add_filter( 'atbdp_listing_card_header_widgets', array( $this, 'add_card_widget' ), 10, 1 );
        add_filter( 'atbdp_listing_card_body_widgets', array( $this, 'add_card_widget' ), 10, 1 );
        add_filter( 'atbdp_listing_card_footer_widgets', array( $this, 'add_card_widget' ), 10, 1 );
        add_filter( 'directorist_listing_card_header_widgets', array( $this, 'add_card_widget' ), 10, 1 );
        add_filter( 'directorist_listing_card_body_widgets', array( $this, 'add_card_widget' ), 10, 1 );
        
        // Grid and list view general
        add_filter( 'atbdp_grid_view_card_widgets', array( $this, 'add_card_widget' ), 10, 1 );
        add_filter( 'atbdp_list_view_card_widgets', array( $this, 'add_card_widget' ), 10, 1 );
        
        // Listings archive widgets
        add_filter( 'directorist_listings_archive_widgets', array( $this, 'add_card_widget' ), 10, 1 );
        add_filter( 'atbdp_listings_archive_widgets', array( $this, 'add_card_widget' ), 10, 1 );
        
        // All listing card builder specific - Directorist 7+ uses these
        add_filter( 'directorist_all_listing_card_widgets', array( $this, 'add_card_widget' ), 10, 1 );
        add_filter( 'directorist_all_listing_layout_widgets', array( $this, 'add_card_widget' ), 10, 1 );
        add_filter( 'atbdp_all_listing_layout_widgets', array( $this, 'add_card_widget' ), 10, 1 );
        
        // ========================================
        // BUILDER GENERAL
        // ========================================
        
        // Builder field settings
        add_filter( 'atbdp_builder_fields', array( $this, 'builder_field_settings' ), 10, 1 );
        add_filter( 'directorist_builder_fields', array( $this, 'builder_field_settings' ), 10, 1 );
        
        // Add to directory builder widgets
        add_filter( 'directorist_builder_available_widgets', array( $this, 'add_available_widget' ), 10, 1 );
        
        // Register for all builder sections
        add_filter( 'atbdp_listing_type_settings_field_list', array( $this, 'add_to_field_list' ), 10, 1 );
        
        // Register widget globally
        add_filter( 'directorist_widgets', array( $this, 'register_global_widget' ), 10, 1 );
        
        // ========================================
        // DIRECTORIST LISTING TYPE CONFIGURATION
        // ========================================
        
        // Listing type default config widgets
        add_filter( 'directorist_listing_type_config_widgets', array( $this, 'add_listing_type_widget' ), 10, 1 );
        add_filter( 'atbdp_listing_type_config_widgets', array( $this, 'add_listing_type_widget' ), 10, 1 );
        
        // Default widgets configuration
        add_filter( 'directorist_default_config_widgets', array( $this, 'add_default_config_widget' ), 10, 1 );
        add_filter( 'atbdp_default_config_widgets', array( $this, 'add_default_config_widget' ), 10, 1 );
        
        // Card preset widgets
        add_filter( 'directorist_card_preset_widgets', array( $this, 'add_card_preset_widget' ), 10, 1 );
        add_filter( 'atbdp_card_preset_widgets', array( $this, 'add_card_preset_widget' ), 10, 1 );
        
        // Listing card sections widgets
        add_filter( 'directorist_listings_card_sections_widgets', array( $this, 'add_card_section_widget' ), 10, 1 );
        add_filter( 'atbdp_listings_card_sections_widgets', array( $this, 'add_card_section_widget' ), 10, 1 );
        
        // Modify builder JSON/script config
        add_filter( 'directorist_builder_script_data', array( $this, 'add_builder_script_data' ), 10, 1 );
        add_filter( 'atbdp_builder_script_data', array( $this, 'add_builder_script_data' ), 10, 1 );
        add_filter( 'directorist_listing_type_form_fields', array( $this, 'add_listing_type_form_fields' ), 10, 1 );
        
        // Add to widget library/registry
        add_filter( 'directorist_widget_library', array( $this, 'add_to_widget_library' ), 10, 1 );
        add_filter( 'atbdp_widget_library', array( $this, 'add_to_widget_library' ), 10, 1 );
        
        // Enqueue our widget registration for the builder
        add_action( 'admin_footer', array( $this, 'output_builder_widget_script' ) );
    }

    /**
     * Register the logo field widget in the form builder
     */
    public function register_field_widget( $widgets ) {
        $widgets['logo'] = $this->get_form_widget_config();
        return $widgets;
    }

    /**
     * Get form widget configuration
     */
    private function get_form_widget_config() {
        return array(
            'label'       => __( 'Logo Image', 'directorist-logo-field' ),
            'icon'        => 'la la-image',
            'widget_name' => 'logo',
            'widget_type' => 'custom',
            'widget_key'  => 'logo',
            'options'     => array(
                'type' => array(
                    'type'  => 'hidden',
                    'value' => 'logo',
                ),
                'widget_name' => array(
                    'type'  => 'hidden',
                    'value' => 'logo',
                ),
                'widget_key' => array(
                    'type'  => 'hidden',
                    'value' => 'logo',
                ),
                'field_key' => array(
                    'type'  => 'hidden',
                    'value' => 'directorist_logo',
                ),
                'label' => array(
                    'type'  => 'text',
                    'label' => __( 'Label', 'directorist-logo-field' ),
                    'value' => __( 'Logo', 'directorist-logo-field' ),
                ),
                'placeholder' => array(
                    'type'  => 'text',
                    'label' => __( 'Placeholder', 'directorist-logo-field' ),
                    'value' => __( 'Click to upload logo', 'directorist-logo-field' ),
                ),
                'required' => array(
                    'type'  => 'toggle',
                    'label' => __( 'Required', 'directorist-logo-field' ),
                    'value' => false,
                ),
                'only_for_admin' => array(
                    'type'  => 'toggle',
                    'label' => __( 'Only For Admin Use', 'directorist-logo-field' ),
                    'value' => false,
                ),
                'max_file_size' => array(
                    'type'  => 'number',
                    'label' => __( 'Max File Size (KB)', 'directorist-logo-field' ),
                    'value' => 2048,
                ),
                'display_width' => array(
                    'type'  => 'number',
                    'label' => __( 'Logo Width (px)', 'directorist-logo-field' ),
                    'value' => 180,
                    'description' => __( 'Height adjusts automatically', 'directorist-logo-field' ),
                ),
            ),
        );
    }

    /**
     * Get display widget configuration for layouts
     */
    private function get_display_widget_config( $context = 'single' ) {
        $width = 'card' === $context ? 100 : 180;
        
        return array(
            'type'        => 'logo',
            'widget_name' => 'logo',
            'widget_key'  => 'logo',
            'label'       => __( 'Logo', 'directorist-logo-field' ),
            'icon'        => 'la la-image',
            'hook'        => 'atbdp_widget_logo',
            'can_move'    => true,
            'can_remove'  => true,
            'options'     => array(
                'display_width' => array(
                    'type'  => 'number',
                    'label' => __( 'Logo Width (px)', 'directorist-logo-field' ),
                    'value' => $width,
                    'description' => __( 'Height adjusts automatically to maintain aspect ratio', 'directorist-logo-field' ),
                ),
                'show_label' => array(
                    'type'  => 'toggle',
                    'label' => __( 'Show Label', 'directorist-logo-field' ),
                    'value' => false,
                ),
            ),
        );
    }

    /**
     * Add submission form widget
     */
    public function add_submission_widget( $widgets ) {
        $widgets['logo'] = array(
            'label' => __( 'Logo Image', 'directorist-logo-field' ),
            'icon'  => 'la la-image',
            'lock'  => false,
            'show'  => true,
        );
        return $widgets;
    }

    /**
     * Register form field configuration
     */
    public function register_form_field( $fields ) {
        $fields['logo'] = array(
            'widget_name' => 'logo',
            'widget_type' => 'custom',
            'widget_key'  => 'logo',
            'type'        => 'logo',
            'label'       => __( 'Logo Image', 'directorist-logo-field' ),
            'icon'        => 'la la-image',
        );
        return $fields;
    }

    /**
     * Render form widget
     */
    public function render_form_widget( $data, $listing_id = 0, $listing = null ) {
        if ( ! $listing_id && $listing ) {
            $listing_id = is_object( $listing ) ? $listing->ID : absint( $listing );
        }
        if ( ! $listing_id && isset( $_GET['edit'] ) ) {
            $listing_id = absint( $_GET['edit'] );
        }
        
        $data['listing_id'] = $listing_id;
        include DIRECTORIST_LOGO_FIELD_PATH . 'templates/form-field-logo.php';
    }

    /**
     * Add to builder available widgets
     */
    public function add_available_widget( $widgets ) {
        $widgets['logo'] = array(
            'label'       => __( 'Logo Image', 'directorist-logo-field' ),
            'icon'        => 'la la-image',
            'widget_name' => 'logo',
            'widget_key'  => 'logo',
        );
        return $widgets;
    }

    /**
     * Add widget to single listing header section
     */
    public function add_single_header_widget( $widgets ) {
        $widgets['logo'] = $this->get_display_widget_config( 'single' );
        return $widgets;
    }

    /**
     * Add widget to single listing content section
     */
    public function add_single_content_widget( $widgets ) {
        $widgets['logo'] = $this->get_display_widget_config( 'single' );
        return $widgets;
    }

    /**
     * Add widget to single listing layout builder (general)
     */
    public function add_single_listing_widget( $widgets ) {
        $widgets['logo'] = $this->get_display_widget_config( 'single' );
        return $widgets;
    }

    /**
     * Register single listing widget with full config
     */
    public function register_single_listing_widget( $fields ) {
        $fields['logo'] = $this->get_display_widget_config( 'single' );
        return $fields;
    }

    /**
     * Add to builder single listing widgets
     */
    public function add_builder_single_widget( $widgets ) {
        $widgets['logo'] = $this->get_display_widget_config( 'single' );
        return $widgets;
    }

    /**
     * Add widget to all listings card layout
     */
    public function add_card_widget( $widgets ) {
        $widgets['logo'] = $this->get_display_widget_config( 'card' );
        return $widgets;
    }

    /**
     * Builder field settings
     */
    public function builder_field_settings( $fields ) {
        $fields['logo'] = array(
            'field_key'   => 'directorist_logo',
            'type'        => 'logo',
            'widget_name' => 'logo',
            'widget_key'  => 'logo',
            'widget'      => 'logo',
            'label'       => __( 'Logo Image', 'directorist-logo-field' ),
            'icon'        => 'la la-image',
        );
        return $fields;
    }

    /**
     * Add to listing type field list
     */
    public function add_to_field_list( $fields ) {
        $fields['logo'] = array(
            'label'       => __( 'Logo Image', 'directorist-logo-field' ),
            'icon'        => 'la la-image',
            'widget_name' => 'logo',
        );
        return $fields;
    }

    /**
     * Add widget to specific section (Directorist 7+ builder system)
     */
    public function add_section_widget( $widgets, $section = '' ) {
        $widgets['logo'] = $this->get_display_widget_config( 'single' );
        return $widgets;
    }

    /**
     * Add to widget groups for builder
     */
    public function add_to_widget_groups( $groups ) {
        // Add logo to common widgets group
        if ( isset( $groups['common'] ) && is_array( $groups['common'] ) ) {
            $groups['common'][] = 'logo';
        }
        
        // Add logo to header widgets group
        if ( isset( $groups['header'] ) && is_array( $groups['header'] ) ) {
            $groups['header'][] = 'logo';
        }
        
        // Add logo to content widgets group
        if ( isset( $groups['content'] ) && is_array( $groups['content'] ) ) {
            $groups['content'][] = 'logo';
        }
        
        return $groups;
    }

    /**
     * Register widget globally
     */
    public function register_global_widget( $widgets ) {
        $widgets['logo'] = array(
            'label'       => __( 'Logo', 'directorist-logo-field' ),
            'icon'        => 'la la-image',
            'widget_name' => 'logo',
            'widget_key'  => 'logo',
            'type'        => 'logo',
            'can_move'    => true,
            'can_remove'  => true,
        );
        return $widgets;
    }

    /**
     * Add widget to listing type configuration
     */
    public function add_listing_type_widget( $widgets ) {
        $widgets['logo'] = array(
            'label'       => __( 'Logo', 'directorist-logo-field' ),
            'icon'        => 'la la-image',
            'widget_name' => 'logo',
            'widget_key'  => 'logo',
            'type'        => 'logo',
            'can_move'    => true,
            'can_remove'  => true,
            'allowIn'     => array( 'avatar', 'title', 'quick_info', 'content', 'footer', 'header', 'body' ),
        );
        return $widgets;
    }

    /**
     * Add to default config widgets
     */
    public function add_default_config_widget( $widgets ) {
        $widgets['logo'] = array(
            'type'        => 'logo',
            'widget_name' => 'logo',
            'widget_key'  => 'logo',
            'label'       => __( 'Logo', 'directorist-logo-field' ),
            'icon'        => 'la la-image',
            'options'     => array(
                'display_width' => array(
                    'type'  => 'number',
                    'label' => __( 'Logo Width (px)', 'directorist-logo-field' ),
                    'value' => 100,
                ),
            ),
        );
        return $widgets;
    }

    /**
     * Add to card preset widgets
     */
    public function add_card_preset_widget( $widgets ) {
        $widgets['logo'] = array(
            'type'        => 'logo',
            'widget_name' => 'logo',
            'widget_key'  => 'logo',
            'label'       => __( 'Logo', 'directorist-logo-field' ),
            'icon'        => 'la la-image',
            'order'       => 5,
            'options'     => array(
                'display_width' => 100,
            ),
        );
        return $widgets;
    }

    /**
     * Add widget to card sections
     */
    public function add_card_section_widget( $sections ) {
        // Add logo to all available sections
        $logo_widget = array(
            'type'        => 'logo',
            'widget_name' => 'logo',
            'widget_key'  => 'logo',
            'label'       => __( 'Logo', 'directorist-logo-field' ),
            'icon'        => 'la la-image',
        );
        
        if ( is_array( $sections ) ) {
            foreach ( $sections as $section_key => $section ) {
                if ( isset( $section['widgets'] ) && is_array( $section['widgets'] ) ) {
                    $sections[ $section_key ]['available_widgets']['logo'] = $logo_widget;
                }
            }
        }
        
        return $sections;
    }

    /**
     * Add to builder script data (JSON config for React builder)
     */
    public function add_builder_script_data( $data ) {
        $logo_config = array(
            'type'        => 'logo',
            'widget_name' => 'logo',
            'widget_key'  => 'logo',
            'label'       => __( 'Logo', 'directorist-logo-field' ),
            'icon'        => 'la la-image',
            'canMove'     => true,
            'canRemove'   => true,
            'options'     => array(
                'display_width' => array(
                    'type'  => 'number',
                    'label' => __( 'Logo Width (px)', 'directorist-logo-field' ),
                    'value' => 100,
                ),
            ),
        );
        
        // Add to various widget registries in the data
        if ( isset( $data['widgets'] ) && is_array( $data['widgets'] ) ) {
            $data['widgets']['logo'] = $logo_config;
        }
        
        if ( isset( $data['available_widgets'] ) && is_array( $data['available_widgets'] ) ) {
            $data['available_widgets']['logo'] = $logo_config;
        }
        
        if ( isset( $data['card_widgets'] ) && is_array( $data['card_widgets'] ) ) {
            $data['card_widgets']['logo'] = $logo_config;
        }
        
        return $data;
    }

    /**
     * Add to listing type form fields
     */
    public function add_listing_type_form_fields( $fields ) {
        $fields['logo'] = array(
            'type'        => 'logo',
            'widget_name' => 'logo',
            'widget_key'  => 'logo',
            'label'       => __( 'Logo', 'directorist-logo-field' ),
            'icon'        => 'la la-image',
        );
        return $fields;
    }

    /**
     * Add to widget library
     */
    public function add_to_widget_library( $library ) {
        $library['logo'] = array(
            'type'        => 'logo',
            'widget_name' => 'logo',
            'widget_key'  => 'logo',
            'label'       => __( 'Logo', 'directorist-logo-field' ),
            'icon'        => 'la la-image',
            'canMove'     => true,
            'canRemove'   => true,
        );
        return $library;
    }

    /**
     * Output JavaScript to register widget in builder
     */
    public function output_builder_widget_script() {
        // Only on Directorist builder pages
        $screen = get_current_screen();
        if ( ! $screen ) {
            return;
        }
        
        $is_builder_page = strpos( $screen->id, 'atbdp' ) !== false 
            || strpos( $screen->id, 'directorist' ) !== false
            || ( isset( $_GET['page'] ) && strpos( $_GET['page'], 'atbdp' ) !== false );
            
        if ( ! $is_builder_page ) {
            return;
        }
        ?>
        <script type="text/javascript">
        (function() {
            // Wait for builder to be ready
            function registerLogoWidget() {
                // Check if Directorist builder globals exist
                if (typeof window.directorist_admin !== 'undefined' && window.directorist_admin.widgets) {
                    window.directorist_admin.widgets.logo = {
                        type: 'logo',
                        widget_name: 'logo',
                        widget_key: 'logo',
                        label: '<?php echo esc_js( __( 'Logo', 'directorist-logo-field' ) ); ?>',
                        icon: 'la la-image',
                        canMove: true,
                        canRemove: true
                    };
                }
                
                // For available widgets
                if (typeof window.directorist_admin !== 'undefined' && window.directorist_admin.available_widgets) {
                    window.directorist_admin.available_widgets.logo = {
                        type: 'logo',
                        widget_name: 'logo',
                        widget_key: 'logo',
                        label: '<?php echo esc_js( __( 'Logo', 'directorist-logo-field' ) ); ?>',
                        icon: 'la la-image'
                    };
                }
                
                // Dispatch event for React to pick up
                var event = new CustomEvent('directorist_widget_registered', { detail: { widget: 'logo' } });
                document.dispatchEvent(event);
            }
            
            // Try immediately and also after DOM ready
            if (document.readyState === 'complete') {
                setTimeout(registerLogoWidget, 500);
            } else {
                document.addEventListener('DOMContentLoaded', function() {
                    setTimeout(registerLogoWidget, 500);
                });
            }
            
            // Also try after a longer delay for React to initialize
            setTimeout(registerLogoWidget, 1500);
            setTimeout(registerLogoWidget, 3000);
        })();
        </script>
        <?php
    }
}
