(function ($) {
  function clamp(value, min, max) {
    return Math.min(max, Math.max(min, value));
  }

  function channelToHex(v) {
    var hex = clamp(parseInt(v, 10) || 0, 0, 255).toString(16);
    return hex.length === 1 ? '0' + hex : hex;
  }

  function rgbToHex(r, g, b) {
    return '#' + channelToHex(r) + channelToHex(g) + channelToHex(b);
  }

  function hexToRgb(hex) {
    var raw = String(hex || '').replace('#', '').trim();

    if (raw.length === 3) {
      return {
        r: parseInt(raw[0] + raw[0], 16),
        g: parseInt(raw[1] + raw[1], 16),
        b: parseInt(raw[2] + raw[2], 16)
      };
    }

    if (raw.length === 6 || raw.length === 8) {
      return {
        r: parseInt(raw.substring(0, 2), 16),
        g: parseInt(raw.substring(2, 4), 16),
        b: parseInt(raw.substring(4, 6), 16)
      };
    }

    return { r: 0, g: 0, b: 0 };
  }

  function parseColorValue(value) {
    var current = String(value || '').trim();
    var parsed = { hex: '#000000', alpha: 100 };

    if (!current) {
      return parsed;
    }

    if (current.toLowerCase() === 'transparent') {
      parsed.alpha = 0;
      return parsed;
    }

    var hexMatch = current.match(/^#([a-f0-9]{3,4}|[a-f0-9]{6}|[a-f0-9]{8})$/i);
    if (hexMatch) {
      var raw = hexMatch[1];
      if (raw.length === 3) {
        parsed.hex = '#' + raw[0] + raw[0] + raw[1] + raw[1] + raw[2] + raw[2];
      } else if (raw.length === 4) {
        parsed.hex = '#' + raw[0] + raw[0] + raw[1] + raw[1] + raw[2] + raw[2];
        parsed.alpha = Math.round((parseInt(raw[3] + raw[3], 16) / 255) * 100);
      } else if (raw.length === 6) {
        parsed.hex = '#' + raw;
      } else {
        parsed.hex = '#' + raw.slice(0, 6);
        parsed.alpha = Math.round((parseInt(raw.slice(6, 8), 16) / 255) * 100);
      }
      return parsed;
    }

    var rgbaMatch = current.match(/^rgba?\(([^)]+)\)$/i);
    if (rgbaMatch) {
      var parts = rgbaMatch[1].split(',').map(function (part) {
        return part.trim();
      });

      if (parts.length >= 3) {
        parsed.hex = rgbToHex(parts[0], parts[1], parts[2]);
      }
      if (parts.length >= 4) {
        parsed.alpha = Math.round(clamp(parseFloat(parts[3]) || 0, 0, 1) * 100);
      }
      return parsed;
    }

    return parsed;
  }

  function toColorValue(hex, alphaPercent) {
    var alpha = clamp((parseInt(alphaPercent, 10) || 0) / 100, 0, 1);
    if (alpha <= 0) {
      return 'transparent';
    }

    if (alpha >= 1) {
      return hex;
    }

    var rgb = hexToRgb(hex);
    return 'rgba(' + rgb.r + ', ' + rgb.g + ', ' + rgb.b + ', ' + alpha.toFixed(2) + ')';
  }

  function initAlphaControl(container) {
    var $container = $(container);
    var $value = $container.find('.rpx-alpha-color-value');
    var $hex = $container.find('.rpx-alpha-color-hex');
    var $alpha = $container.find('.rpx-alpha-color-alpha');
    var $alphaOut = $container.find('.rpx-alpha-color-alpha-output');
    var $transparent = $container.find('.rpx-alpha-transparent');

    if (!$value.length || !$hex.length || !$alpha.length) {
      return;
    }

    var initial = parseColorValue($value.val());
    $hex.val(initial.hex);
    $alpha.val(initial.alpha);
    $alphaOut.text(initial.alpha + '%');

    function syncToSetting() {
      var color = toColorValue($hex.val(), $alpha.val());
      $value.val(color).trigger('change');
      $alphaOut.text($alpha.val() + '%');
    }

    $hex.on('input change', syncToSetting);
    $alpha.on('input change', syncToSetting);

    $transparent.on('click', function () {
      $alpha.val(0);
      syncToSetting();
    });
  }

  $(function () {
    $('.rpx-alpha-color-control').each(function () {
      initAlphaControl(this);
    });

    if (window.wp && window.wp.customize && window.wp.customize.control) {
      window.wp.customize.control.each(function (control) {
        if (control.container && control.container.find('.rpx-alpha-color-control').length) {
          control.deferred.embedded.done(function () {
            control.container.find('.rpx-alpha-color-control').each(function () {
              initAlphaControl(this);
            });
          });
        }
      });
    }
  });
})(jQuery);
