(function () {
  var INIT_FLAG = '__rpxGalleryCarouselInit';
  var AUTO_DELAY_MS = 4200;

  if (window[INIT_FLAG]) {
    return;
  }
  window[INIT_FLAG] = true;

  function smoothScrollTo(grid, left) {
    if (typeof grid.scrollTo === 'function') {
      try {
        grid.scrollTo({ left: left, behavior: 'smooth' });
        return;
      } catch (e) {
        // Ignore unsupported smooth options and fall back.
      }
    }

    grid.scrollLeft = left;
  }

  function maxScrollLeft(grid) {
    return Math.max(0, grid.scrollWidth - grid.clientWidth);
  }

  function getPageWidth(grid) {
    return Math.max(1, Math.round(grid.clientWidth));
  }

  function moveGrid(grid, direction) {
    var max = maxScrollLeft(grid);
    if (max <= 1) {
      return;
    }

    var page = getPageWidth(grid);
    var current = Math.round(grid.scrollLeft);
    var target = current + page * direction;

    if (direction > 0 && current >= max - 2) {
      target = 0;
    } else if (direction < 0 && current <= 2) {
      target = max;
    } else {
      target = Math.max(0, Math.min(target, max));
    }

    smoothScrollTo(grid, target);
  }

  function updateControlState(grid, prev, next) {
    var canScroll = maxScrollLeft(grid) > 1;
    prev.disabled = !canScroll;
    next.disabled = !canScroll;
    prev.setAttribute('aria-disabled', canScroll ? 'false' : 'true');
    next.setAttribute('aria-disabled', canScroll ? 'false' : 'true');
  }

  function bindCarousel(section) {
    if (section.getAttribute('data-rpx-carousel-bound') === 'true') {
      return;
    }

    var grid = section.querySelector('[data-rpx-gallery-carousel]');
    var prev = section.querySelector('[data-rpx-carousel-prev]');
    var next = section.querySelector('[data-rpx-carousel-next]');
    if (!grid || !prev || !next) {
      return;
    }

    var prefersReducedMotion = window.matchMedia && window.matchMedia('(prefers-reduced-motion: reduce)').matches;
    var autoEnabled = grid.getAttribute('data-rpx-carousel-autoplay') === 'true' && !prefersReducedMotion;
    var autoTimer = null;

    function stopAuto() {
      if (!autoTimer) {
        return;
      }
      window.clearInterval(autoTimer);
      autoTimer = null;
    }

    function startAuto() {
      if (!autoEnabled || autoTimer || maxScrollLeft(grid) <= 1) {
        return;
      }

      autoTimer = window.setInterval(function () {
        moveGrid(grid, 1);
      }, AUTO_DELAY_MS);
    }

    prev.addEventListener('click', function () {
      moveGrid(grid, -1);
      stopAuto();
      startAuto();
    });

    next.addEventListener('click', function () {
      moveGrid(grid, 1);
      stopAuto();
      startAuto();
    });

    section.addEventListener('mouseenter', stopAuto);
    section.addEventListener('mouseleave', startAuto);
    section.addEventListener('focusin', stopAuto);
    section.addEventListener('focusout', function (event) {
      if (!section.contains(event.relatedTarget)) {
        startAuto();
      }
    });

    window.addEventListener('resize', function () {
      updateControlState(grid, prev, next);
      stopAuto();
      startAuto();
    });

    updateControlState(grid, prev, next);
    startAuto();

    section.setAttribute('data-rpx-carousel-bound', 'true');
  }

  function initGalleryCarousel() {
    document.querySelectorAll('.rpx-gallery-section.is-carousel').forEach(bindCarousel);
  }

  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', initGalleryCarousel);
  } else {
    initGalleryCarousel();
  }

  window.addEventListener('pageshow', initGalleryCarousel);
})();
