(function () {
  var DEFAULT_AUTOPLAY_MS = 6200;
  var DEFAULT_TRANSITION_MS = 760;
  var INIT_FLAG = 'rpxSliderInit';

  function toArray(collection) {
    return Array.prototype.slice.call(collection || []);
  }

  function clamp(value, min, max) {
    return Math.min(Math.max(value, min), max);
  }

  function parseInteger(value, fallback) {
    var parsed = parseInt(value, 10);
    return isNaN(parsed) ? fallback : parsed;
  }

  function getSliderConfig(root) {
    var effect = (root.dataset.rpxSliderEffect || 'slide').toLowerCase();
    if (effect !== 'fade' && effect !== 'slide') {
      effect = 'slide';
    }

    var durationSeconds = clamp(parseInteger(root.dataset.rpxSliderDuration, 6), 2, 30);
    var autoplayMs = durationSeconds * 1000;
    var transitionMs = clamp(parseInteger(root.dataset.rpxSliderTransition, DEFAULT_TRANSITION_MS), 150, 4000);

    return {
      effect: effect,
      autoplayMs: autoplayMs > 0 ? autoplayMs : DEFAULT_AUTOPLAY_MS,
      transitionMs: transitionMs > 0 ? transitionMs : DEFAULT_TRANSITION_MS
    };
  }

  function getRootSlides(root) {
    return toArray(root.children).filter(function (node) {
      return node.classList && node.classList.contains('rpx-hero-slide');
    });
  }

  function disableControls(prev, next) {
    if (prev) {
      prev.disabled = true;
      prev.setAttribute('aria-hidden', 'true');
    }
    if (next) {
      next.disabled = true;
      next.setAttribute('aria-hidden', 'true');
    }
  }

  function setActiveSlide(realSlides, activeIndex) {
    realSlides.forEach(function (slide, index) {
      var isActive = index === activeIndex;
      slide.classList.toggle('is-active', isActive);
      slide.setAttribute('aria-hidden', isActive ? 'false' : 'true');
    });
  }

  function initFadeSlider(root, realSlides, prev, next, config) {
    var initialRealIndex = realSlides.findIndex(function (slide) {
      return slide.classList.contains('is-active');
    });
    if (initialRealIndex < 0) {
      initialRealIndex = 0;
    }

    realSlides.forEach(function (slide) {
      slide.style.transitionDuration = config.transitionMs + 'ms';
      slide.style.transitionTimingFunction = 'ease-in-out';
    });

    var current = initialRealIndex;
    var isAnimating = false;
    var animationTimer = null;
    var autoplayTimer = null;

    function clearAnimationTimer() {
      if (animationTimer) {
        window.clearTimeout(animationTimer);
        animationTimer = null;
      }
      isAnimating = false;
    }

    function move(direction, restartAutoplay) {
      if (isAnimating) {
        return;
      }

      isAnimating = true;
      var nextIndex = current + direction;
      if (nextIndex < 0) {
        nextIndex = realSlides.length - 1;
      } else if (nextIndex >= realSlides.length) {
        nextIndex = 0;
      }
      setActiveSlide(realSlides, nextIndex);
      current = nextIndex;

      animationTimer = window.setTimeout(clearAnimationTimer, config.transitionMs + 80);

      if (restartAutoplay) {
        startAutoplay();
      }
    }

    function stopAutoplay() {
      if (autoplayTimer) {
        window.clearInterval(autoplayTimer);
        autoplayTimer = null;
      }
    }

    function startAutoplay() {
      stopAutoplay();
      autoplayTimer = window.setInterval(function () {
        move(1, false);
      }, config.autoplayMs);
    }

    if (prev) {
      prev.disabled = false;
      prev.removeAttribute('aria-hidden');
      prev.addEventListener('click', function () {
        move(-1, true);
      });
    }

    if (next) {
      next.disabled = false;
      next.removeAttribute('aria-hidden');
      next.addEventListener('click', function () {
        move(1, true);
      });
    }

    document.addEventListener('visibilitychange', function () {
      if (document.hidden) {
        stopAutoplay();
      } else {
        startAutoplay();
      }
    });

    setActiveSlide(realSlides, current);
    startAutoplay();
  }

  function initSlideSlider(root, realSlides, prev, next, config) {
    var initialRealIndex = realSlides.findIndex(function (slide) {
      return slide.classList.contains('is-active');
    });
    if (initialRealIndex < 0) {
      initialRealIndex = 0;
    }

    var firstClone = realSlides[0].cloneNode(true);
    var lastClone = realSlides[realSlides.length - 1].cloneNode(true);
    firstClone.classList.remove('is-active');
    lastClone.classList.remove('is-active');
    firstClone.setAttribute('aria-hidden', 'true');
    lastClone.setAttribute('aria-hidden', 'true');

    var track = document.createElement('div');
    track.className = 'rpx-hero-track';
    track.appendChild(lastClone);
    realSlides.forEach(function (slide) {
      track.appendChild(slide);
    });
    track.appendChild(firstClone);
    root.insertBefore(track, root.firstChild);

    var trackIndex = initialRealIndex + 1;
    var isAnimating = false;
    var animationSafetyTimer = null;
    var autoplayTimer = null;

    function realIndexFromTrack(trackPos) {
      var total = realSlides.length;
      var realIndex = (trackPos - 1) % total;
      return realIndex < 0 ? realIndex + total : realIndex;
    }

    function applyTransform(skipTransition) {
      if (skipTransition) {
        track.style.transition = 'none';
      } else {
        track.style.transition = 'transform ' + config.transitionMs + 'ms ease-in-out';
      }
      track.style.transform = 'translate3d(' + (-trackIndex * 100) + '%, 0, 0)';

      if (skipTransition) {
        // Flush layout before restoring transition so the jump is instant.
        track.offsetHeight;
        track.style.transition = 'transform ' + config.transitionMs + 'ms ease-in-out';
      }
    }

    function syncActive() {
      setActiveSlide(realSlides, realIndexFromTrack(trackIndex));
    }

    function clearAnimationState() {
      if (animationSafetyTimer) {
        window.clearTimeout(animationSafetyTimer);
        animationSafetyTimer = null;
      }
      isAnimating = false;
    }

    function finalizeLoopPosition() {
      var changed = false;
      if (trackIndex === 0) {
        trackIndex = realSlides.length;
        changed = true;
      } else if (trackIndex === realSlides.length + 1) {
        trackIndex = 1;
        changed = true;
      }

      if (changed) {
        applyTransform(true);
      }

      syncActive();
      clearAnimationState();
    }

    function move(direction, restartAutoplay) {
      if (isAnimating) {
        return;
      }

      isAnimating = true;
      trackIndex += direction;
      applyTransform(false);
      syncActive();

      animationSafetyTimer = window.setTimeout(function () {
        finalizeLoopPosition();
      }, config.transitionMs + 120);

      if (restartAutoplay) {
        startAutoplay();
      }
    }

    function stopAutoplay() {
      if (autoplayTimer) {
        window.clearInterval(autoplayTimer);
        autoplayTimer = null;
      }
    }

    function startAutoplay() {
      stopAutoplay();
      autoplayTimer = window.setInterval(function () {
        move(1, false);
      }, config.autoplayMs);
    }

    track.addEventListener('transitionend', function (event) {
      if (event && event.target !== track) {
        return;
      }
      finalizeLoopPosition();
    });

    if (prev) {
      prev.disabled = false;
      prev.removeAttribute('aria-hidden');
      prev.addEventListener('click', function () {
        move(-1, true);
      });
    }

    if (next) {
      next.disabled = false;
      next.removeAttribute('aria-hidden');
      next.addEventListener('click', function () {
        move(1, true);
      });
    }

    document.addEventListener('visibilitychange', function () {
      if (document.hidden) {
        stopAutoplay();
      } else {
        startAutoplay();
      }
    });

    syncActive();
    applyTransform(true);
    startAutoplay();
  }

  function initSlider(root) {
    if (!root || root.dataset[INIT_FLAG] === 'true') {
      return;
    }

    var realSlides = getRootSlides(root);
    var prev = root.querySelector('[data-rpx-prev]');
    var next = root.querySelector('[data-rpx-next]');
    var config = getSliderConfig(root);

    if (realSlides.length < 2) {
      disableControls(prev, next);
      root.dataset[INIT_FLAG] = 'true';
      return;
    }

    if (config.effect === 'fade') {
      initFadeSlider(root, realSlides, prev, next, config);
    } else {
      initSlideSlider(root, realSlides, prev, next, config);
    }

    root.dataset[INIT_FLAG] = 'true';
  }

  function initAllSliders() {
    toArray(document.querySelectorAll('[data-rpx-slider]')).forEach(initSlider);
  }

  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', initAllSliders);
  } else {
    initAllSliders();
  }

  window.addEventListener('load', initAllSliders);
  window.addEventListener('pageshow', initAllSliders);
})();
