<?php
/**
 * Theme customizer settings.
 *
 * @package Ravenpine
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Default values for theme mods.
 *
 * @param string $key Theme mod key.
 * @return mixed
 */
function ravenpine_theme_mod_default( $key ) {
	$defaults = array(
		'ravenpine_hero_heading'              => __( 'Seasonal food. Warm nights. One table for everyone.', 'ravenpine' ),
		'ravenpine_hero_text'                 => __( 'Craft cocktails, wood-fired specials, and neighborhood hospitality in one place.', 'ravenpine' ),
		'ravenpine_hero_button_label'         => __( 'Book A Table', 'ravenpine' ),
		'ravenpine_hero_button_url'           => '#location-hours',
		'ravenpine_menu_section_eyebrow'      => __( 'Menus', 'ravenpine' ),
		'ravenpine_menu_section_heading'      => __( 'Explore What We Are Serving', 'ravenpine' ),
		'ravenpine_menu_section_alignment'    => 'left',
		'ravenpine_gallery_section_eyebrow'   => __( 'Gallery', 'ravenpine' ),
		'ravenpine_gallery_section_heading'   => __( 'From Our Kitchen', 'ravenpine' ),
		'ravenpine_gallery_randomize'         => 0,
		'ravenpine_gallery_use_carousel'      => 0,
		'ravenpine_hero_cards_enabled'        => 1,
		'ravenpine_hero_slider_controls_enabled' => 1,
		'ravenpine_hero_slider_effect'        => 'slide',
		'ravenpine_hero_slider_duration'      => 6,
		'ravenpine_hero_slider_transition'    => 760,
		'ravenpine_hero_card_bg_color'        => '#1f1a16',
		'ravenpine_hero_card_border_color'    => '#ffffff',
		'ravenpine_hero_card_text_color'      => '#ffffff',
		'ravenpine_hero_image_overlay_color'  => 'rgba(0, 0, 0, 0.35)',
		'ravenpine_hero_button_bg_color'      => '#b4542d',
		'ravenpine_hero_button_text_color'    => '#ffffff',
		'ravenpine_heading_font'              => 'playfair',
		'ravenpine_body_font'                 => 'source-sans',
		'ravenpine_font_upload_helper'        => '',
		'ravenpine_accent_color'              => '#b4542d',
		'ravenpine_background_color'          => '#f6f2ea',
		'ravenpine_background_image'          => '',
		'ravenpine_foreground_color'          => '#1f1a16',
		'ravenpine_heading_color'             => '#1f1a16',
		'ravenpine_surface_color'             => '#ffffff',
		'ravenpine_nav_bg_color'              => '#f6f2ea',
		'ravenpine_nav_link_color'            => '#1f1a16',
		'ravenpine_menu_button_bg_color'      => '#f4e3d8',
		'ravenpine_menu_button_text_color'    => '#1f1a16',
		'ravenpine_menu_button_border_color'  => '#b4542d',
		'ravenpine_menu_section_bg_color'      => 'transparent',
		'ravenpine_menu_section_bg_image'      => '',
		'ravenpine_about_section_bg_color'     => 'transparent',
		'ravenpine_about_section_bg_image'     => '',
		'ravenpine_about_section_heading_color' => '#1f1a16',
		'ravenpine_about_section_text_color'    => '#1f1a16',
		'ravenpine_gallery_section_bg_color'   => 'transparent',
		'ravenpine_gallery_section_bg_image'   => '',
		'ravenpine_gallery_section_text_color' => '#1f1a16',
		'ravenpine_location_section_bg_color'  => 'transparent',
		'ravenpine_location_section_bg_image'  => '',
		'ravenpine_social_section_bg_color'    => 'transparent',
		'ravenpine_social_section_bg_image'    => '',
		'ravenpine_menu_section_heading_color' => '#1f1a16',
		'ravenpine_menu_section_text_color'    => '#1f1a16',
		'ravenpine_location_card_bg_color'     => '#ffffff',
		'ravenpine_section_border_top_enabled' => 0,
		'ravenpine_section_border_bottom_enabled' => 0,
		'ravenpine_section_border_width'       => 1,
		'ravenpine_section_border_color'       => 'rgba(31, 26, 22, 0.16)',
		'ravenpine_footer_bg_color'            => 'transparent',
		'ravenpine_footer_bg_image'            => '',
		'ravenpine_footer_logo'                => '',
		'ravenpine_footer_text_color'          => '#1f1a16',
		'ravenpine_tagline_color'              => '#1f1a16',
		'ravenpine_section_padding'           => 88,
		'ravenpine_social_facebook'           => '',
		'ravenpine_social_instagram'          => '',
		'ravenpine_social_x'                  => '',
		'ravenpine_social_tiktok'             => '',
		'ravenpine_social_yelp'               => '',
	);

	return $defaults[ $key ] ?? '';
}

/**
 * Built-in font choices shipped with the theme.
 *
 * @return array
 */
function ravenpine_get_base_font_choices() {
	return array(
		'playfair'      => array(
			'label'  => 'Playfair Display',
			'stack'  => '"Playfair Display", Georgia, serif',
			'google' => 'Playfair+Display:wght@500;700',
		),
		'source-sans'   => array(
			'label'  => 'Source Sans 3',
			'stack'  => '"Source Sans 3", -apple-system, BlinkMacSystemFont, "Segoe UI", sans-serif',
			'google' => 'Source+Sans+3:wght@400;600;700',
		),
		'lora'          => array(
			'label'  => 'Lora',
			'stack'  => '"Lora", Georgia, serif',
			'google' => 'Lora:wght@400;600;700',
		),
		'merriweather'  => array(
			'label'  => 'Merriweather',
			'stack'  => '"Merriweather", Georgia, serif',
			'google' => 'Merriweather:wght@400;700',
		),
		'montserrat'    => array(
			'label'  => 'Montserrat',
			'stack'  => '"Montserrat", "Segoe UI", sans-serif',
			'google' => 'Montserrat:wght@400;600;700',
		),
		'nunito'        => array(
			'label'  => 'Nunito Sans',
			'stack'  => '"Nunito Sans", "Segoe UI", sans-serif',
			'google' => 'Nunito+Sans:wght@400;600;700',
		),
		'oswald'        => array(
			'label'  => 'Oswald',
			'stack'  => '"Oswald", "Segoe UI", sans-serif',
			'google' => 'Oswald:wght@400;500;700',
		),
		'system-sans'   => array(
			'label'  => 'System Sans',
			'stack'  => '-apple-system, BlinkMacSystemFont, "Segoe UI", sans-serif',
			'google' => '',
		),
		'system-serif'  => array(
			'label'  => 'System Serif',
			'stack'  => 'Georgia, "Times New Roman", serif',
			'google' => '',
		),
	);
}

/**
 * Font choices used by customizer controls.
 *
 * @param bool $include_uploaded Include uploaded fonts.
 * @return array
 */
function ravenpine_get_font_choices( $include_uploaded = true ) {
	$choices = ravenpine_get_base_font_choices();
	if ( $include_uploaded ) {
		$choices = array_merge( $choices, ravenpine_get_uploaded_font_choices() );
	}

	return $choices;
}

/**
 * Build readable font metadata from an uploaded file name.
 *
 * @param string $filename Font file name without extension.
 * @return array
 */
function ravenpine_font_metadata_from_filename( $filename ) {
	$filename = trim( (string) $filename );
	$pretty   = preg_replace( '/(?<!^)([A-Z])/', ' $1', str_replace( array( '-', '_' ), ' ', $filename ) );
	$pretty   = trim( preg_replace( '/\s+/', ' ', (string) $pretty ) );
	$pretty   = '' !== $pretty ? $pretty : 'Uploaded Font';

	$family_part = preg_split( '/[-_]/', $filename );
	$family_part = is_array( $family_part ) && ! empty( $family_part[0] ) ? $family_part[0] : $filename;
	$family_part = preg_replace( '/(?<!^)([A-Z])/', ' $1', (string) $family_part );
	$family_name = ravenpine_sanitize_font_family( $family_part );

	return array(
		'label'  => $pretty,
		'family' => 'RavenpineFont ' . absint( crc32( $filename ) ) . ' ' . $family_name,
		'style'  => 'normal',
		'weight' => '100 900',
	);
}

/**
 * Collect uploaded fonts from the Media Library.
 *
 * @return array
 */
function ravenpine_get_uploaded_font_choices() {
	static $choices = null;
	if ( null !== $choices ) {
		return $choices;
	}

	$cache_key = 'ravenpine_uploaded_font_choices_v1';
	$cached    = get_transient( $cache_key );
	if ( false !== $cached && is_array( $cached ) ) {
		$choices = $cached;
		return $choices;
	}

	$choices = array();

	$attachment_ids = get_posts(
		array(
			'post_type'      => 'attachment',
			'post_status'    => 'inherit',
			'posts_per_page' => -1,
			'orderby'        => 'date',
			'order'          => 'DESC',
			'fields'         => 'ids',
			'meta_query'     => array(
				'relation' => 'OR',
				array(
					'key'     => '_wp_attached_file',
					'value'   => '.ttf',
					'compare' => 'LIKE',
				),
				array(
					'key'     => '_wp_attached_file',
					'value'   => '.otf',
					'compare' => 'LIKE',
				),
				array(
					'key'     => '_wp_attached_file',
					'value'   => '.woff',
					'compare' => 'LIKE',
				),
				array(
					'key'     => '_wp_attached_file',
					'value'   => '.woff2',
					'compare' => 'LIKE',
				),
			),
			'no_found_rows'          => true,
			'update_post_meta_cache' => true,
			'update_post_term_cache' => false,
			'cache_results'          => true,
		)
	);

	$allowed_ext = array( 'ttf', 'otf', 'woff', 'woff2' );

	foreach ( $attachment_ids as $attachment_id ) {
		$file_path = get_attached_file( $attachment_id );
		$url       = wp_get_attachment_url( $attachment_id );
		if ( empty( $file_path ) || empty( $url ) ) {
			continue;
		}

		$ext = strtolower( pathinfo( $file_path, PATHINFO_EXTENSION ) );
		if ( ! in_array( $ext, $allowed_ext, true ) ) {
			continue;
		}

		$filename = pathinfo( $file_path, PATHINFO_FILENAME );
		$meta     = ravenpine_font_metadata_from_filename( $filename );
		$key      = 'uploaded-' . absint( $attachment_id );

		$choices[ $key ] = array(
			'label'    => $meta['label'] . ' (Uploaded)',
			'stack'    => '"' . ravenpine_sanitize_font_family( $meta['family'] ) . '", -apple-system, BlinkMacSystemFont, "Segoe UI", sans-serif',
			'google'   => '',
			'uploaded' => true,
			'url'      => ravenpine_force_https_url( $url ),
			'family'   => ravenpine_sanitize_font_family( $meta['family'] ),
			'style'    => $meta['style'],
			'weight'   => $meta['weight'],
		);
	}

	set_transient( $cache_key, $choices, DAY_IN_SECONDS );

	return $choices;
}

/**
 * Clear cached uploaded font choices.
 *
 * @param int $attachment_id Attachment post ID.
 * @return void
 */
function ravenpine_clear_uploaded_font_choices_cache( $attachment_id = 0 ) {
	$attachment_id = absint( $attachment_id );
	if ( $attachment_id > 0 && 'attachment' !== get_post_type( $attachment_id ) ) {
		return;
	}

	delete_transient( 'ravenpine_uploaded_font_choices_v1' );
}
add_action( 'add_attachment', 'ravenpine_clear_uploaded_font_choices_cache' );
add_action( 'edit_attachment', 'ravenpine_clear_uploaded_font_choices_cache' );
add_action( 'delete_attachment', 'ravenpine_clear_uploaded_font_choices_cache' );
add_action( 'customize_save_after', 'ravenpine_clear_uploaded_font_choices_cache' );

/**
 * Sanitize checkbox controls.
 *
 * @param mixed $checked Value to sanitize.
 * @return int
 */
function ravenpine_sanitize_checkbox( $checked ) {
	return ( ! empty( $checked ) ) ? 1 : 0;
}

/**
 * Sanitize section alignment setting.
 *
 * @param string $value Alignment value.
 * @return string
 */
function ravenpine_sanitize_section_alignment( $value ) {
	$allowed = array( 'left', 'center', 'right' );
	return in_array( $value, $allowed, true ) ? $value : 'left';
}

/**
 * Sanitize hero slider effect setting.
 *
 * @param string $value Effect value.
 * @return string
 */
function ravenpine_sanitize_hero_slider_effect( $value ) {
	$allowed = array( 'slide', 'fade' );
	return in_array( (string) $value, $allowed, true ) ? (string) $value : 'slide';
}

/**
 * Sanitize hero slider duration (seconds).
 *
 * @param mixed $value Duration value.
 * @return int
 */
function ravenpine_sanitize_hero_slider_duration( $value ) {
	$value = absint( $value );
	return min( max( $value, 2 ), 30 );
}

/**
 * Sanitize hero slider transition time (milliseconds).
 *
 * @param mixed $value Transition value.
 * @return int
 */
function ravenpine_sanitize_hero_slider_transition( $value ) {
	$value = absint( $value );
	return min( max( $value, 150 ), 4000 );
}

/**
 * Check if a font key references an uploaded font.
 *
 * @param string $font_key Font key.
 * @return bool
 */
function ravenpine_is_uploaded_font_key( $font_key ) {
	return 0 === strpos( (string) $font_key, 'uploaded-' );
}

/**
 * Sanitize font choice controls.
 *
 * @param string $value Selected value.
 * @return string
 */
function ravenpine_sanitize_font_choice( $value ) {
	$value        = (string) $value;
	$base_choices = ravenpine_get_base_font_choices();

	if ( isset( $base_choices[ $value ] ) ) {
		return $value;
	}

	if ( ravenpine_is_uploaded_font_key( $value ) ) {
		$uploaded = ravenpine_get_uploaded_font_choices();
		if ( isset( $uploaded[ $value ] ) ) {
			return $value;
		}
	}

	return ravenpine_theme_mod_default( 'ravenpine_body_font' );
}

/**
 * Sanitize custom font family text.
 *
 * @param string $value Input family name.
 * @return string
 */
function ravenpine_sanitize_font_family( $value ) {
	$value = sanitize_text_field( $value );
	$value = preg_replace( '/[^A-Za-z0-9 _-]/', '', $value );
	$value = trim( (string) $value );
	return '' !== $value ? $value : 'RavenpineCustomFont';
}

/**
 * Sanitize CSS color-like values.
 * Supports hex, rgb/rgba, hsl/hsla, and transparent.
 *
 * @param string $value Raw value.
 * @return string
 */
function ravenpine_sanitize_css_color( $value ) {
	$value = trim( (string) $value );
	if ( '' === $value ) {
		return '';
	}

	if ( 'transparent' === strtolower( $value ) ) {
		return 'transparent';
	}

	if ( preg_match( '/^#(?:[A-Fa-f0-9]{3}|[A-Fa-f0-9]{4}|[A-Fa-f0-9]{6}|[A-Fa-f0-9]{8})$/', $value ) ) {
		return $value;
	}

	if ( preg_match( '/^(?:rgb|rgba|hsl|hsla)\(\s*[-0-9.,% ]+\s*\)$/i', $value ) ) {
		return $value;
	}

	return '';
}

/**
 * Sanitize values inserted into inline CSS.
 *
 * @param string $value Value.
 * @return string
 */
function ravenpine_sanitize_css_value( $value ) {
	return (string) preg_replace( '/[{};<>`]/', '', (string) $value );
}

/**
 * Determine whether frontend asset URLs should be upgraded to HTTPS.
 *
 * This is intentionally tolerant of reverse proxies and staging setups where
 * WP's canonical URLs may still be HTTP while the request is HTTPS.
 *
 * @return bool
 */
function ravenpine_should_force_https_assets() {
	if ( is_ssl() ) {
		return true;
	}

	if ( isset( $_SERVER['HTTPS'] ) && 'off' !== strtolower( (string) $_SERVER['HTTPS'] ) && '' !== (string) $_SERVER['HTTPS'] ) {
		return true;
	}

	if ( isset( $_SERVER['HTTP_X_FORWARDED_PROTO'] ) && false !== stripos( (string) $_SERVER['HTTP_X_FORWARDED_PROTO'], 'https' ) ) {
		return true;
	}

	if ( function_exists( 'wp_is_using_https' ) && wp_is_using_https() ) {
		return true;
	}

	$home_scheme = wp_parse_url( home_url(), PHP_URL_SCHEME );
	return 'https' === strtolower( (string) $home_scheme );
}

/**
 * Upgrade a URL to HTTPS when current context indicates secure output.
 *
 * @param string $url URL to normalize.
 * @return string
 */
function ravenpine_force_https_url( $url ) {
	$url = esc_url_raw( (string) $url );
	if ( '' === $url ) {
		return '';
	}

	if ( ravenpine_should_force_https_assets() ) {
		return set_url_scheme( $url, 'https' );
	}

	return $url;
}

/**
 * Normalize insecure URLs inside arbitrary CSS text.
 *
 * @param string $css Raw CSS text.
 * @return string
 */
function ravenpine_force_https_in_css( $css ) {
	$css = (string) $css;
	if ( '' === $css || ! ravenpine_should_force_https_assets() ) {
		return $css;
	}

	$css = str_replace( 'http://ravenpinefull.noirkitchen.com/', 'https://ravenpinefull.noirkitchen.com/', $css );
	$css = str_replace( "url('http://", "url('https://", $css );
	$css = str_replace( 'url("http://', 'url("https://', $css );
	$css = str_replace( 'url(http://', 'url(https://', $css );

	return $css;
}

/**
 * Upgrade insecure font/CSS URLs saved in Additional CSS.
 *
 * @param string $css         Additional CSS.
 * @param string $stylesheet  Stylesheet slug.
 * @return string
 */
function ravenpine_filter_wp_get_custom_css( $css, $stylesheet ) {
	unset( $stylesheet );
	return ravenpine_force_https_in_css( $css );
}
add_filter( 'wp_get_custom_css', 'ravenpine_filter_wp_get_custom_css', 20, 2 );

/**
 * Recursively force HTTPS URLs inside theme.json user data.
 *
 * @param mixed $value Any theme.json node.
 * @return mixed
 */
function ravenpine_force_https_in_theme_json_value( $value ) {
	if ( is_array( $value ) ) {
		foreach ( $value as $key => $item ) {
			$value[ $key ] = ravenpine_force_https_in_theme_json_value( $item );
		}
		return $value;
	}

	if ( is_string( $value ) ) {
		return ravenpine_force_https_in_css( $value );
	}

	return $value;
}

/**
 * Upgrade insecure URLs saved in user Global Styles data.
 *
 * @param WP_Theme_JSON_Data $theme_json Theme JSON data container.
 * @return WP_Theme_JSON_Data
 */
function ravenpine_filter_theme_json_user_data_https( $theme_json ) {
	if ( ! ravenpine_should_force_https_assets() ) {
		return $theme_json;
	}

	if ( ! is_object( $theme_json ) || ! method_exists( $theme_json, 'get_data' ) || ! method_exists( $theme_json, 'update_with' ) ) {
		return $theme_json;
	}

	$data = $theme_json->get_data();
	if ( ! is_array( $data ) || empty( $data ) ) {
		return $theme_json;
	}

	$updated = ravenpine_force_https_in_theme_json_value( $data );
	if ( $updated !== $data ) {
		$theme_json->update_with( $updated );
	}

	return $theme_json;
}
add_filter( 'wp_theme_json_data_user', 'ravenpine_filter_theme_json_user_data_https' );

/**
 * Parse saved color value for control defaults.
 *
 * @param string $value Saved color value.
 * @return array
 */
function ravenpine_parse_color_control_value( $value ) {
	$value = trim( (string) $value );
	$hex   = '#000000';
	$alpha = 100;

	if ( '' === $value ) {
		return array(
			'hex'   => $hex,
			'alpha' => $alpha,
		);
	}

	if ( 'transparent' === strtolower( $value ) ) {
		return array(
			'hex'   => $hex,
			'alpha' => 0,
		);
	}

	if ( preg_match( '/^#([A-Fa-f0-9]{3}|[A-Fa-f0-9]{4}|[A-Fa-f0-9]{6}|[A-Fa-f0-9]{8})$/', $value, $matches ) ) {
		$raw = $matches[1];
		if ( 3 === strlen( $raw ) ) {
			$hex = '#' . $raw[0] . $raw[0] . $raw[1] . $raw[1] . $raw[2] . $raw[2];
		} elseif ( 4 === strlen( $raw ) ) {
			$hex   = '#' . $raw[0] . $raw[0] . $raw[1] . $raw[1] . $raw[2] . $raw[2];
			$alpha = (int) round( hexdec( $raw[3] . $raw[3] ) / 255 * 100 );
		} elseif ( 6 === strlen( $raw ) ) {
			$hex = '#' . $raw;
		} else {
			$hex   = '#' . substr( $raw, 0, 6 );
			$alpha = (int) round( hexdec( substr( $raw, 6, 2 ) ) / 255 * 100 );
		}

		return array(
			'hex'   => strtolower( $hex ),
			'alpha' => max( 0, min( 100, $alpha ) ),
		);
	}

	if ( preg_match( '/^rgba?\(([^)]+)\)$/i', $value, $matches ) ) {
		$parts = array_map( 'trim', explode( ',', $matches[1] ) );
		if ( count( $parts ) >= 3 ) {
			$r = max( 0, min( 255, (int) $parts[0] ) );
			$g = max( 0, min( 255, (int) $parts[1] ) );
			$b = max( 0, min( 255, (int) $parts[2] ) );
			$hex = sprintf( '#%02x%02x%02x', $r, $g, $b );
		}
		if ( count( $parts ) >= 4 ) {
			$alpha = (int) round( max( 0, min( 1, (float) $parts[3] ) ) * 100 );
		}
	}

	return array(
		'hex'   => strtolower( $hex ),
		'alpha' => max( 0, min( 100, $alpha ) ),
	);
}

if ( class_exists( 'WP_Customize_Control' ) && ! class_exists( 'Ravenpine_Alpha_Color_Control' ) ) {
	/**
	 * Custom alpha color control for Customizer.
	 */
	class Ravenpine_Alpha_Color_Control extends WP_Customize_Control {
		/**
		 * Control type.
		 *
		 * @var string
		 */
		public $type = 'ravenpine_alpha_color';

		/**
		 * Render control content.
		 *
		 * @return void
		 */
		public function render_content() {
			$parsed = ravenpine_parse_color_control_value( $this->value() );
			?>
			<div class="rpx-alpha-color-control">
				<?php if ( ! empty( $this->label ) ) : ?>
					<span class="customize-control-title"><?php echo esc_html( $this->label ); ?></span>
				<?php endif; ?>
				<?php if ( ! empty( $this->description ) ) : ?>
					<span class="description customize-control-description"><?php echo esc_html( $this->description ); ?></span>
				<?php endif; ?>
				<input type="hidden" class="rpx-alpha-color-value" <?php $this->link(); ?> value="<?php echo esc_attr( $this->value() ); ?>" />
				<div class="rpx-alpha-color-row">
					<input type="color" class="rpx-alpha-color-hex" value="<?php echo esc_attr( $parsed['hex'] ); ?>" />
					<button type="button" class="button button-secondary rpx-alpha-transparent"><?php esc_html_e( 'Transparent', 'ravenpine' ); ?></button>
				</div>
				<div class="rpx-alpha-color-row">
					<input type="range" class="rpx-alpha-color-alpha" min="0" max="100" step="1" value="<?php echo esc_attr( (string) $parsed['alpha'] ); ?>" />
					<span class="rpx-alpha-color-alpha-output"><?php echo esc_html( $parsed['alpha'] . '%' ); ?></span>
				</div>
			</div>
			<?php
		}
	}
}

/**
 * Enqueue customizer control assets.
 */
function ravenpine_enqueue_customizer_control_assets() {
	$version = wp_get_theme()->get( 'Version' );
	wp_enqueue_script(
		'ravenpine-customizer-alpha-control',
		get_template_directory_uri() . '/assets/js/customizer-alpha-color.js',
		array( 'jquery', 'customize-controls' ),
		$version,
		true
	);
	wp_enqueue_style(
		'ravenpine-customizer-alpha-control',
		get_template_directory_uri() . '/assets/css/customizer-alpha-color.css',
		array(),
		$version
	);
}
add_action( 'customize_controls_enqueue_scripts', 'ravenpine_enqueue_customizer_control_assets' );

/**
 * Resolve a font key into a stack string.
 *
 * @param string $font_key Font key.
 * @param string $context  heading|body
 * @return string
 */
function ravenpine_get_font_stack( $font_key, $context = 'body' ) {
	$font_key     = (string) $font_key;
	$base_choices = ravenpine_get_base_font_choices();

	if ( isset( $base_choices[ $font_key ]['stack'] ) ) {
		return $base_choices[ $font_key ]['stack'];
	}

	if ( ravenpine_is_uploaded_font_key( $font_key ) ) {
		$uploaded_choices = ravenpine_get_uploaded_font_choices();
		if ( isset( $uploaded_choices[ $font_key ]['stack'] ) ) {
			return $uploaded_choices[ $font_key ]['stack'];
		}
	}

	$fallback = ( 'heading' === $context ) ? ravenpine_theme_mod_default( 'ravenpine_heading_font' ) : ravenpine_theme_mod_default( 'ravenpine_body_font' );
	return isset( $base_choices[ $fallback ]['stack'] ) ? $base_choices[ $fallback ]['stack'] : $base_choices['source-sans']['stack'];
}

/**
 * Build an @font-face rule when a custom font file is provided.
 *
 * @param string $url    Font file URL.
 * @param string $family Font family name.
 * @param string $style  Font style.
 * @param string $weight Font weight.
 * @return string
 */
function ravenpine_build_font_face_css( $url, $family, $style = 'normal', $weight = '400' ) {
	$url    = ravenpine_force_https_url( $url );
	$family = ravenpine_sanitize_font_family( $family );
	if ( empty( $url ) || empty( $family ) ) {
		return '';
	}

	$ext    = strtolower( pathinfo( wp_parse_url( $url, PHP_URL_PATH ), PATHINFO_EXTENSION ) );
	$format = 'truetype';
	$weight = trim( (string) $weight );
	if ( ! preg_match( '/^\d{3}( \d{3})?$/', $weight ) ) {
		$weight = '100 900';
	}
	$style  = in_array( $style, array( 'normal', 'italic', 'oblique' ), true ) ? $style : 'normal';
	if ( 'woff2' === $ext ) {
		$format = 'woff2';
	} elseif ( 'woff' === $ext ) {
		$format = 'woff';
	} elseif ( 'otf' === $ext ) {
		$format = 'opentype';
	}

	return sprintf(
		'@font-face{font-family:"%1$s";src:url("%2$s") format("%3$s");font-style:%4$s;font-weight:%5$s;font-display:swap;}',
		ravenpine_sanitize_css_value( $family ),
		esc_url_raw( $url ),
		ravenpine_sanitize_css_value( $format ),
		ravenpine_sanitize_css_value( $style ),
		$weight
	);
}

/**
 * Get a Google Fonts URL for selected heading/body fonts.
 *
 * @return string
 */
function ravenpine_get_google_fonts_url() {
	$heading_key = (string) get_theme_mod( 'ravenpine_heading_font', ravenpine_theme_mod_default( 'ravenpine_heading_font' ) );
	$body_key    = (string) get_theme_mod( 'ravenpine_body_font', ravenpine_theme_mod_default( 'ravenpine_body_font' ) );
	$choices     = ravenpine_get_base_font_choices();

	if ( ! isset( $choices[ $heading_key ] ) && ! ravenpine_is_uploaded_font_key( $heading_key ) ) {
		$heading_key = ravenpine_theme_mod_default( 'ravenpine_heading_font' );
	}
	if ( ! isset( $choices[ $body_key ] ) && ! ravenpine_is_uploaded_font_key( $body_key ) ) {
		$body_key = ravenpine_theme_mod_default( 'ravenpine_body_font' );
	}

	$families = array();
	if ( isset( $choices[ $heading_key ] ) && ! empty( $choices[ $heading_key ]['google'] ) ) {
		$families[] = $choices[ $heading_key ]['google'];
	}
	if ( isset( $choices[ $body_key ] ) && ! empty( $choices[ $body_key ]['google'] ) ) {
		$families[] = $choices[ $body_key ]['google'];
	}

	$families = array_unique( $families );
	if ( empty( $families ) ) {
		return '';
	}

	return 'https://fonts.googleapis.com/css2?family=' . implode( '&family=', $families ) . '&display=swap';
}

/**
 * Register customizer settings.
 *
 * @param WP_Customize_Manager $wp_customize Customizer object.
 */
function ravenpine_customize_register( $wp_customize ) {
	$wp_customize->add_section(
		'ravenpine_hero',
		array(
			'title'    => __( 'Hero Overlay Content', 'ravenpine' ),
			'priority' => 35,
		)
	);

	$wp_customize->add_setting(
		'ravenpine_hero_heading',
		array(
			'default'           => ravenpine_theme_mod_default( 'ravenpine_hero_heading' ),
			'sanitize_callback' => 'sanitize_text_field',
		)
	);
	$wp_customize->add_control(
		'ravenpine_hero_heading',
		array(
			'label'   => __( 'Heading', 'ravenpine' ),
			'section' => 'ravenpine_hero',
			'type'    => 'text',
		)
	);

	$wp_customize->add_setting(
		'ravenpine_hero_text',
		array(
			'default'           => ravenpine_theme_mod_default( 'ravenpine_hero_text' ),
			'sanitize_callback' => 'sanitize_textarea_field',
		)
	);
	$wp_customize->add_control(
		'ravenpine_hero_text',
		array(
			'label'   => __( 'Supporting Text', 'ravenpine' ),
			'section' => 'ravenpine_hero',
			'type'    => 'textarea',
		)
	);

	$wp_customize->add_setting(
		'ravenpine_hero_button_label',
		array(
			'default'           => ravenpine_theme_mod_default( 'ravenpine_hero_button_label' ),
			'sanitize_callback' => 'sanitize_text_field',
		)
	);
	$wp_customize->add_control(
		'ravenpine_hero_button_label',
		array(
			'label'   => __( 'Button Label', 'ravenpine' ),
			'section' => 'ravenpine_hero',
			'type'    => 'text',
		)
	);

	$wp_customize->add_setting(
		'ravenpine_hero_button_url',
		array(
			'default'           => ravenpine_theme_mod_default( 'ravenpine_hero_button_url' ),
			'sanitize_callback' => 'esc_url_raw',
		)
	);
	$wp_customize->add_control(
		'ravenpine_hero_button_url',
		array(
			'label'   => __( 'Button URL', 'ravenpine' ),
			'section' => 'ravenpine_hero',
			'type'    => 'url',
		)
	);

	$wp_customize->add_setting(
		'ravenpine_hero_cards_enabled',
		array(
			'default'           => ravenpine_theme_mod_default( 'ravenpine_hero_cards_enabled' ),
			'sanitize_callback' => 'ravenpine_sanitize_checkbox',
		)
	);
	$wp_customize->add_control(
		'ravenpine_hero_cards_enabled',
		array(
			'label'   => __( 'Show colored cards behind logo and text', 'ravenpine' ),
			'section' => 'ravenpine_hero',
			'type'    => 'checkbox',
		)
	);

	$wp_customize->add_setting(
		'ravenpine_hero_slider_controls_enabled',
		array(
			'default'           => ravenpine_theme_mod_default( 'ravenpine_hero_slider_controls_enabled' ),
			'sanitize_callback' => 'ravenpine_sanitize_checkbox',
		)
	);
	$wp_customize->add_control(
		'ravenpine_hero_slider_controls_enabled',
		array(
			'label'   => __( 'Show slider arrow buttons', 'ravenpine' ),
			'section' => 'ravenpine_hero',
			'type'    => 'checkbox',
		)
	);

	$wp_customize->add_setting(
		'ravenpine_hero_slider_effect',
		array(
			'default'           => ravenpine_theme_mod_default( 'ravenpine_hero_slider_effect' ),
			'sanitize_callback' => 'ravenpine_sanitize_hero_slider_effect',
		)
	);
	$wp_customize->add_control(
		'ravenpine_hero_slider_effect',
		array(
			'label'   => __( 'Slider Effect', 'ravenpine' ),
			'section' => 'ravenpine_hero',
			'type'    => 'select',
			'choices' => array(
				'slide' => __( 'Slide', 'ravenpine' ),
				'fade'  => __( 'Fade', 'ravenpine' ),
			),
		)
	);

	$wp_customize->add_setting(
		'ravenpine_hero_slider_duration',
		array(
			'default'           => ravenpine_theme_mod_default( 'ravenpine_hero_slider_duration' ),
			'sanitize_callback' => 'ravenpine_sanitize_hero_slider_duration',
		)
	);
	$wp_customize->add_control(
		'ravenpine_hero_slider_duration',
		array(
			'label'       => __( 'Slide Duration (seconds)', 'ravenpine' ),
			'description' => __( 'How long each slide stays visible before advancing.', 'ravenpine' ),
			'section'     => 'ravenpine_hero',
			'type'        => 'number',
			'input_attrs' => array(
				'min'  => 2,
				'max'  => 30,
				'step' => 1,
			),
		)
	);

	$wp_customize->add_setting(
		'ravenpine_hero_slider_transition',
		array(
			'default'           => ravenpine_theme_mod_default( 'ravenpine_hero_slider_transition' ),
			'sanitize_callback' => 'ravenpine_sanitize_hero_slider_transition',
		)
	);
	$wp_customize->add_control(
		'ravenpine_hero_slider_transition',
		array(
			'label'       => __( 'Transition Time (ms)', 'ravenpine' ),
			'description' => __( 'How long the fade/slide animation takes.', 'ravenpine' ),
			'section'     => 'ravenpine_hero',
			'type'        => 'number',
			'input_attrs' => array(
				'min'  => 150,
				'max'  => 4000,
				'step' => 50,
			),
		)
	);

	$wp_customize->add_section(
		'ravenpine_menu_section',
		array(
			'title'    => __( 'Menu Section', 'ravenpine' ),
			'priority' => 36,
		)
	);

	$wp_customize->add_setting(
		'ravenpine_menu_section_eyebrow',
		array(
			'default'           => ravenpine_theme_mod_default( 'ravenpine_menu_section_eyebrow' ),
			'sanitize_callback' => 'sanitize_text_field',
		)
	);
	$wp_customize->add_control(
		'ravenpine_menu_section_eyebrow',
		array(
			'label'   => __( 'Eyebrow Text', 'ravenpine' ),
			'section' => 'ravenpine_menu_section',
			'type'    => 'text',
		)
	);

	$wp_customize->add_setting(
		'ravenpine_menu_section_heading',
		array(
			'default'           => ravenpine_theme_mod_default( 'ravenpine_menu_section_heading' ),
			'sanitize_callback' => 'sanitize_text_field',
		)
	);
	$wp_customize->add_control(
		'ravenpine_menu_section_heading',
		array(
			'label'   => __( 'Section Heading', 'ravenpine' ),
			'section' => 'ravenpine_menu_section',
			'type'    => 'text',
		)
	);

	$wp_customize->add_setting(
		'ravenpine_menu_section_alignment',
		array(
			'default'           => ravenpine_theme_mod_default( 'ravenpine_menu_section_alignment' ),
			'sanitize_callback' => 'ravenpine_sanitize_section_alignment',
		)
	);
	$wp_customize->add_control(
		'ravenpine_menu_section_alignment',
		array(
			'label'   => __( 'Section Alignment', 'ravenpine' ),
			'section' => 'ravenpine_menu_section',
			'type'    => 'select',
			'choices' => array(
				'left'   => __( 'Left', 'ravenpine' ),
				'center' => __( 'Center', 'ravenpine' ),
				'right'  => __( 'Right', 'ravenpine' ),
			),
		)
	);

	$wp_customize->add_section(
		'ravenpine_gallery_section',
		array(
			'title'    => __( 'Gallery Section', 'ravenpine' ),
			'priority' => 37,
		)
	);

	$wp_customize->add_setting(
		'ravenpine_gallery_section_eyebrow',
		array(
			'default'           => ravenpine_theme_mod_default( 'ravenpine_gallery_section_eyebrow' ),
			'sanitize_callback' => 'sanitize_text_field',
		)
	);
	$wp_customize->add_control(
		'ravenpine_gallery_section_eyebrow',
		array(
			'label'   => __( 'Eyebrow Text', 'ravenpine' ),
			'section' => 'ravenpine_gallery_section',
			'type'    => 'text',
		)
	);

	$wp_customize->add_setting(
		'ravenpine_gallery_section_heading',
		array(
			'default'           => ravenpine_theme_mod_default( 'ravenpine_gallery_section_heading' ),
			'sanitize_callback' => 'sanitize_text_field',
		)
	);
	$wp_customize->add_control(
		'ravenpine_gallery_section_heading',
		array(
			'label'   => __( 'Section Heading', 'ravenpine' ),
			'section' => 'ravenpine_gallery_section',
			'type'    => 'text',
		)
	);

	$wp_customize->add_setting(
		'ravenpine_gallery_randomize',
		array(
			'default'           => ravenpine_theme_mod_default( 'ravenpine_gallery_randomize' ),
			'sanitize_callback' => 'ravenpine_sanitize_checkbox',
		)
	);
	$wp_customize->add_control(
		'ravenpine_gallery_randomize',
		array(
			'label'       => __( 'Randomize Image Order', 'ravenpine' ),
			'section'     => 'ravenpine_gallery_section',
			'type'        => 'checkbox',
			'description' => __( 'When enabled, gallery images are shuffled on each page load.', 'ravenpine' ),
		)
	);

	$wp_customize->add_setting(
		'ravenpine_gallery_use_carousel',
		array(
			'default'           => ravenpine_theme_mod_default( 'ravenpine_gallery_use_carousel' ),
			'sanitize_callback' => 'ravenpine_sanitize_checkbox',
		)
	);
	$wp_customize->add_control(
		'ravenpine_gallery_use_carousel',
		array(
			'label'       => __( 'Display As Carousel', 'ravenpine' ),
			'section'     => 'ravenpine_gallery_section',
			'type'        => 'checkbox',
			'description' => __( 'Shows the gallery as a horizontal carousel with navigation controls.', 'ravenpine' ),
		)
	);

	$wp_customize->add_section(
		'ravenpine_typography',
		array(
			'title'    => __( 'Typography', 'ravenpine' ),
			'priority' => 38,
		)
	);

	$font_choices = array();
	foreach ( ravenpine_get_font_choices() as $key => $font ) {
		$font_choices[ $key ] = $font['label'];
	}

	$wp_customize->add_setting(
		'ravenpine_heading_font',
		array(
			'default'           => ravenpine_theme_mod_default( 'ravenpine_heading_font' ),
			'sanitize_callback' => 'ravenpine_sanitize_font_choice',
		)
	);
	$wp_customize->add_control(
		'ravenpine_heading_font',
		array(
			'label'   => __( 'Heading Font', 'ravenpine' ),
			'section' => 'ravenpine_typography',
			'type'    => 'select',
			'choices' => $font_choices,
		)
	);

	$wp_customize->add_setting(
		'ravenpine_body_font',
		array(
			'default'           => ravenpine_theme_mod_default( 'ravenpine_body_font' ),
			'sanitize_callback' => 'ravenpine_sanitize_font_choice',
		)
	);
	$wp_customize->add_control(
		'ravenpine_body_font',
		array(
			'label'   => __( 'Body Font', 'ravenpine' ),
			'section' => 'ravenpine_typography',
			'type'    => 'select',
			'choices' => $font_choices,
		)
	);

	$wp_customize->add_setting(
		'ravenpine_font_upload_helper',
		array(
			'default'           => ravenpine_theme_mod_default( 'ravenpine_font_upload_helper' ),
			'sanitize_callback' => 'esc_url_raw',
		)
	);
	$wp_customize->add_control(
		new WP_Customize_Upload_Control(
			$wp_customize,
			'ravenpine_font_upload_helper',
			array(
				'label'       => __( 'Upload Font File (WOFF2/WOFF/TTF/OTF)', 'ravenpine' ),
				'section'     => 'ravenpine_typography',
				'description' => __( 'Uploaded fonts appear in Heading/Body font dropdowns after saving and refreshing the Customizer.', 'ravenpine' ),
			)
		)
	);

	$wp_customize->add_panel(
		'ravenpine_style_panel',
		array(
			'title'       => __( 'Theme Style Controls', 'ravenpine' ),
			'priority'    => 40,
			'description' => __( 'Styles are organized by block/area so each section is easier to manage.', 'ravenpine' ),
		)
	);

	$style_sections = array(
		'ravenpine_style_global'   => array(
			'title'       => __( 'Global / Page', 'ravenpine' ),
			'description' => __( 'Site-wide page and typography-adjacent color settings.', 'ravenpine' ),
			'priority'    => 10,
		),
		'ravenpine_style_header'   => array(
			'title'    => __( 'Header', 'ravenpine' ),
			'priority' => 20,
		),
		'ravenpine_style_hero'     => array(
			'title'    => __( 'Hero Slider Block', 'ravenpine' ),
			'priority' => 30,
		),
		'ravenpine_style_menu'     => array(
			'title'    => __( 'Menu Buttons Block', 'ravenpine' ),
			'priority' => 40,
		),
		'ravenpine_style_about'    => array(
			'title'    => __( 'About Us Block', 'ravenpine' ),
			'priority' => 50,
		),
		'ravenpine_style_gallery'  => array(
			'title'    => __( 'Gallery Block', 'ravenpine' ),
			'priority' => 60,
		),
		'ravenpine_style_location' => array(
			'title'    => __( 'Location & Hours Block', 'ravenpine' ),
			'priority' => 70,
		),
		'ravenpine_style_social'   => array(
			'title'    => __( 'Social Links Block', 'ravenpine' ),
			'priority' => 80,
		),
		'ravenpine_style_footer'   => array(
			'title'    => __( 'Footer', 'ravenpine' ),
			'priority' => 90,
		),
		'ravenpine_style_borders'  => array(
			'title'    => __( 'Section Borders & Spacing', 'ravenpine' ),
			'priority' => 100,
		),
	);

	foreach ( $style_sections as $section_id => $section_args ) {
		$wp_customize->add_section(
			$section_id,
			array(
				'title'       => $section_args['title'],
				'description' => $section_args['description'] ?? '',
				'priority'    => $section_args['priority'],
				'panel'       => 'ravenpine_style_panel',
			)
		);
	}

	$bg_image_controls = array(
		'ravenpine_background_image'         => array(
			'label'       => __( 'Fixed Page Background Image', 'ravenpine' ),
			'section'     => 'ravenpine_style_global',
			'description' => __( 'This image fills the page background and stays fixed while content scrolls.', 'ravenpine' ),
		),
		'ravenpine_menu_section_bg_image'    => array(
			'label'       => __( 'Menu Section Background Image', 'ravenpine' ),
			'section'     => 'ravenpine_style_menu',
			'description' => __( 'Optional image shown behind the Menu section.', 'ravenpine' ),
		),
		'ravenpine_about_section_bg_image'   => array(
			'label'       => __( 'About Section Background Image', 'ravenpine' ),
			'section'     => 'ravenpine_style_about',
			'description' => __( 'Optional image shown behind the About Us section.', 'ravenpine' ),
		),
		'ravenpine_gallery_section_bg_image' => array(
			'label'       => __( 'Gallery Section Background Image', 'ravenpine' ),
			'section'     => 'ravenpine_style_gallery',
			'description' => __( 'Optional image shown behind the Gallery section.', 'ravenpine' ),
		),
		'ravenpine_location_section_bg_image' => array(
			'label'       => __( 'Location & Hours Background Image', 'ravenpine' ),
			'section'     => 'ravenpine_style_location',
			'description' => __( 'Optional image shown behind the Location & Hours section.', 'ravenpine' ),
		),
		'ravenpine_social_section_bg_image'  => array(
			'label'       => __( 'Social Links Background Image', 'ravenpine' ),
			'section'     => 'ravenpine_style_social',
			'description' => __( 'Optional image shown behind the Social Links section.', 'ravenpine' ),
		),
		'ravenpine_footer_bg_image'          => array(
			'label'       => __( 'Footer Background Image', 'ravenpine' ),
			'section'     => 'ravenpine_style_footer',
			'description' => __( 'Optional image shown behind the footer.', 'ravenpine' ),
		),
	);

	foreach ( $bg_image_controls as $setting_id => $control_args ) {
		$wp_customize->add_setting(
			$setting_id,
			array(
				'default'           => ravenpine_theme_mod_default( $setting_id ),
				'sanitize_callback' => 'esc_url_raw',
			)
		);
		$wp_customize->add_control(
			new WP_Customize_Image_Control(
				$wp_customize,
				$setting_id,
				array(
					'label'       => $control_args['label'],
					'section'     => $control_args['section'],
					'description' => $control_args['description'],
				)
			)
		);
	}

	$wp_customize->add_setting(
		'ravenpine_footer_logo',
		array(
			'default'           => ravenpine_theme_mod_default( 'ravenpine_footer_logo' ),
			'sanitize_callback' => 'esc_url_raw',
		)
	);
	$wp_customize->add_control(
		new WP_Customize_Image_Control(
			$wp_customize,
			'ravenpine_footer_logo',
			array(
				'label'       => __( 'Footer Logo Override', 'ravenpine' ),
				'section'     => 'title_tagline',
				'description' => __( 'Optional footer-specific logo. If empty, the main site logo is used.', 'ravenpine' ),
			)
		)
	);

	$color_controls = array(
		'ravenpine_accent_color'             => array(
			'label'   => __( 'Accent Color', 'ravenpine' ),
			'section' => 'ravenpine_style_global',
		),
		'ravenpine_background_color'         => array(
			'label'   => __( 'Background Color', 'ravenpine' ),
			'section' => 'ravenpine_style_global',
		),
		'ravenpine_foreground_color'         => array(
			'label'   => __( 'Body Text Color', 'ravenpine' ),
			'section' => 'ravenpine_style_global',
		),
		'ravenpine_heading_color'            => array(
			'label'   => __( 'Heading Color', 'ravenpine' ),
			'section' => 'ravenpine_style_global',
		),
		'ravenpine_surface_color'            => array(
			'label'   => __( 'Card Surface Color', 'ravenpine' ),
			'section' => 'ravenpine_style_global',
		),
		'ravenpine_nav_bg_color'             => array(
			'label'   => __( 'Header Background Color', 'ravenpine' ),
			'section' => 'ravenpine_style_header',
		),
		'ravenpine_nav_link_color'           => array(
			'label'   => __( 'Header Link Color', 'ravenpine' ),
			'section' => 'ravenpine_style_header',
		),
		'ravenpine_hero_image_overlay_color' => array(
			'label'   => __( 'Hero Image Overlay Color', 'ravenpine' ),
			'section' => 'ravenpine_style_hero',
		),
		'ravenpine_hero_card_bg_color'       => array(
			'label'   => __( 'Hero Card Background Color', 'ravenpine' ),
			'section' => 'ravenpine_style_hero',
		),
		'ravenpine_hero_card_border_color'   => array(
			'label'   => __( 'Hero Card Border Color', 'ravenpine' ),
			'section' => 'ravenpine_style_hero',
		),
		'ravenpine_hero_card_text_color'     => array(
			'label'   => __( 'Hero Card Text Color', 'ravenpine' ),
			'section' => 'ravenpine_style_hero',
		),
		'ravenpine_hero_button_bg_color'     => array(
			'label'   => __( 'Hero Button Background Color', 'ravenpine' ),
			'section' => 'ravenpine_style_hero',
		),
		'ravenpine_hero_button_text_color'   => array(
			'label'   => __( 'Hero Button Text Color', 'ravenpine' ),
			'section' => 'ravenpine_style_hero',
		),
		'ravenpine_menu_button_bg_color'     => array(
			'label'   => __( 'Menu Button Background Color', 'ravenpine' ),
			'section' => 'ravenpine_style_menu',
		),
		'ravenpine_menu_button_text_color'   => array(
			'label'   => __( 'Menu Button Text Color', 'ravenpine' ),
			'section' => 'ravenpine_style_menu',
		),
		'ravenpine_menu_button_border_color' => array(
			'label'   => __( 'Menu Button Border Color', 'ravenpine' ),
			'section' => 'ravenpine_style_menu',
		),
		'ravenpine_menu_section_bg_color'    => array(
			'label'   => __( 'Menu Section Background Color', 'ravenpine' ),
			'section' => 'ravenpine_style_menu',
		),
		'ravenpine_menu_section_heading_color' => array(
			'label'   => __( 'Menu Section Heading Color', 'ravenpine' ),
			'section' => 'ravenpine_style_menu',
		),
		'ravenpine_menu_section_text_color'  => array(
			'label'   => __( 'Menu Section Text Color', 'ravenpine' ),
			'section' => 'ravenpine_style_menu',
		),
		'ravenpine_about_section_bg_color'   => array(
			'label'   => __( 'About Section Background Color', 'ravenpine' ),
			'section' => 'ravenpine_style_about',
		),
		'ravenpine_gallery_section_bg_color' => array(
			'label'   => __( 'Gallery Section Background Color', 'ravenpine' ),
			'section' => 'ravenpine_style_gallery',
		),
		'ravenpine_gallery_section_text_color' => array(
			'label'   => __( 'Gallery Section Text Color', 'ravenpine' ),
			'section' => 'ravenpine_style_gallery',
		),
		'ravenpine_about_section_heading_color' => array(
			'label'   => __( 'About Heading Color', 'ravenpine' ),
			'section' => 'ravenpine_style_about',
		),
		'ravenpine_about_section_text_color' => array(
			'label'   => __( 'About Text Color', 'ravenpine' ),
			'section' => 'ravenpine_style_about',
		),
		'ravenpine_location_section_bg_color' => array(
			'label'   => __( 'Location & Hours Background Color', 'ravenpine' ),
			'section' => 'ravenpine_style_location',
		),
		'ravenpine_location_card_bg_color'   => array(
			'label'   => __( 'Location/Hours Card Background Color', 'ravenpine' ),
			'section' => 'ravenpine_style_location',
		),
		'ravenpine_social_section_bg_color'  => array(
			'label'   => __( 'Social Links Background Color', 'ravenpine' ),
			'section' => 'ravenpine_style_social',
		),
		'ravenpine_footer_bg_color'          => array(
			'label'   => __( 'Footer Background Color', 'ravenpine' ),
			'section' => 'ravenpine_style_footer',
		),
		'ravenpine_footer_text_color'        => array(
			'label'   => __( 'Footer Text Color', 'ravenpine' ),
			'section' => 'ravenpine_style_footer',
		),
		'ravenpine_tagline_color'            => array(
			'label'   => __( 'Footer Tagline Color', 'ravenpine' ),
			'section' => 'ravenpine_style_footer',
		),
		'ravenpine_section_border_color'     => array(
			'label'   => __( 'Section Border Color', 'ravenpine' ),
			'section' => 'ravenpine_style_borders',
		),
	);

	foreach ( $color_controls as $setting_id => $args ) {
		$wp_customize->add_setting(
			$setting_id,
			array(
				'default'           => ravenpine_theme_mod_default( $setting_id ),
				'sanitize_callback' => 'ravenpine_sanitize_css_color',
			)
		);
		if ( class_exists( 'Ravenpine_Alpha_Color_Control' ) ) {
			$wp_customize->add_control(
				new Ravenpine_Alpha_Color_Control(
					$wp_customize,
					$setting_id,
					array(
						'label'       => $args['label'],
						'section'     => $args['section'],
						'description' => __( 'Pick a color and set opacity.', 'ravenpine' ),
					)
				)
			);
		} else {
			$wp_customize->add_control(
				$setting_id,
				array(
					'label'       => $args['label'],
					'section'     => $args['section'],
					'type'        => 'text',
					'description' => __( 'Use #hex, rgba(...), or transparent.', 'ravenpine' ),
				)
			);
		}
	}

	$wp_customize->add_setting(
		'ravenpine_section_padding',
		array(
			'default'           => ravenpine_theme_mod_default( 'ravenpine_section_padding' ),
			'sanitize_callback' => 'absint',
		)
	);
	$wp_customize->add_control(
		'ravenpine_section_padding',
		array(
			'label'       => __( 'Default Section Vertical Padding (px)', 'ravenpine' ),
			'section'     => 'ravenpine_style_borders',
			'type'        => 'number',
			'input_attrs' => array(
				'min'  => 24,
				'max'  => 220,
				'step' => 4,
			),
		)
	);

	$wp_customize->add_setting(
		'ravenpine_section_border_top_enabled',
		array(
			'default'           => ravenpine_theme_mod_default( 'ravenpine_section_border_top_enabled' ),
			'sanitize_callback' => 'ravenpine_sanitize_checkbox',
		)
	);
	$wp_customize->add_control(
		'ravenpine_section_border_top_enabled',
		array(
			'label'   => __( 'Show Top Border On Sections', 'ravenpine' ),
			'section' => 'ravenpine_style_borders',
			'type'    => 'checkbox',
		)
	);

	$wp_customize->add_setting(
		'ravenpine_section_border_bottom_enabled',
		array(
			'default'           => ravenpine_theme_mod_default( 'ravenpine_section_border_bottom_enabled' ),
			'sanitize_callback' => 'ravenpine_sanitize_checkbox',
		)
	);
	$wp_customize->add_control(
		'ravenpine_section_border_bottom_enabled',
		array(
			'label'   => __( 'Show Bottom Border On Sections', 'ravenpine' ),
			'section' => 'ravenpine_style_borders',
			'type'    => 'checkbox',
		)
	);

	$wp_customize->add_setting(
		'ravenpine_section_border_width',
		array(
			'default'           => ravenpine_theme_mod_default( 'ravenpine_section_border_width' ),
			'sanitize_callback' => 'absint',
		)
	);
	$wp_customize->add_control(
		'ravenpine_section_border_width',
		array(
			'label'       => __( 'Section Border Width (px)', 'ravenpine' ),
			'section'     => 'ravenpine_style_borders',
			'type'        => 'number',
			'input_attrs' => array(
				'min'  => 0,
				'max'  => 12,
				'step' => 1,
			),
		)
	);

	$wp_customize->add_section(
		'ravenpine_social',
		array(
			'title'       => __( 'Social Links', 'ravenpine' ),
			'priority'    => 45,
			'description' => __( 'Links are shown by the Social Links block, typically near the bottom of the one-page layout.', 'ravenpine' ),
		)
	);

	$social_fields = array(
		'ravenpine_social_facebook'  => __( 'Facebook URL', 'ravenpine' ),
		'ravenpine_social_instagram' => __( 'Instagram URL', 'ravenpine' ),
		'ravenpine_social_x'         => __( 'X URL', 'ravenpine' ),
		'ravenpine_social_tiktok'    => __( 'TikTok URL', 'ravenpine' ),
		'ravenpine_social_yelp'      => __( 'Yelp URL', 'ravenpine' ),
	);

	foreach ( $social_fields as $setting_id => $label ) {
		$wp_customize->add_setting(
			$setting_id,
			array(
				'default'           => ravenpine_theme_mod_default( $setting_id ),
				'sanitize_callback' => 'esc_url_raw',
			)
		);
		$wp_customize->add_control(
			$setting_id,
			array(
				'label'   => $label,
				'section' => 'ravenpine_social',
				'type'    => 'url',
			)
		);
	}
}
add_action( 'customize_register', 'ravenpine_customize_register' );

/**
 * Build CSS vars from customizer values.
 *
 * @return string
 */
function ravenpine_get_customizer_css_vars() {
	$color_defaults = array(
		'ravenpine_accent_color',
		'ravenpine_background_color',
		'ravenpine_foreground_color',
		'ravenpine_heading_color',
		'ravenpine_surface_color',
		'ravenpine_nav_bg_color',
		'ravenpine_nav_link_color',
		'ravenpine_hero_image_overlay_color',
		'ravenpine_hero_card_bg_color',
		'ravenpine_hero_card_border_color',
		'ravenpine_hero_card_text_color',
		'ravenpine_hero_button_bg_color',
		'ravenpine_hero_button_text_color',
		'ravenpine_menu_button_bg_color',
		'ravenpine_menu_button_text_color',
		'ravenpine_menu_button_border_color',
		'ravenpine_menu_section_bg_color',
		'ravenpine_about_section_bg_color',
		'ravenpine_about_section_heading_color',
		'ravenpine_about_section_text_color',
		'ravenpine_gallery_section_bg_color',
		'ravenpine_gallery_section_text_color',
		'ravenpine_location_section_bg_color',
		'ravenpine_social_section_bg_color',
		'ravenpine_menu_section_heading_color',
		'ravenpine_menu_section_text_color',
		'ravenpine_location_card_bg_color',
		'ravenpine_section_border_color',
		'ravenpine_footer_bg_color',
		'ravenpine_footer_text_color',
		'ravenpine_tagline_color',
	);

	$colors = array();
	foreach ( $color_defaults as $setting_id ) {
		$value             = ravenpine_sanitize_css_color( get_theme_mod( $setting_id, ravenpine_theme_mod_default( $setting_id ) ) );
		$colors[ $setting_id ] = '' !== $value ? $value : ravenpine_theme_mod_default( $setting_id );
	}

	$pad              = absint( get_theme_mod( 'ravenpine_section_padding', ravenpine_theme_mod_default( 'ravenpine_section_padding' ) ) );
	$pad              = min( max( $pad, 24 ), 220 );
	$border_width     = absint( get_theme_mod( 'ravenpine_section_border_width', ravenpine_theme_mod_default( 'ravenpine_section_border_width' ) ) );
	$border_width     = min( max( $border_width, 0 ), 12 );
	$border_top       = (bool) get_theme_mod( 'ravenpine_section_border_top_enabled', ravenpine_theme_mod_default( 'ravenpine_section_border_top_enabled' ) );
	$border_bottom    = (bool) get_theme_mod( 'ravenpine_section_border_bottom_enabled', ravenpine_theme_mod_default( 'ravenpine_section_border_bottom_enabled' ) );
	$border_top_width = $border_top ? $border_width : 0;
	$border_bottom_width = $border_bottom ? $border_width : 0;
	$heading_font_key = ravenpine_sanitize_font_choice( get_theme_mod( 'ravenpine_heading_font', ravenpine_theme_mod_default( 'ravenpine_heading_font' ) ) );
	$body_font_key    = ravenpine_sanitize_font_choice( get_theme_mod( 'ravenpine_body_font', ravenpine_theme_mod_default( 'ravenpine_body_font' ) ) );
	$heading_font     = ravenpine_get_font_stack( $heading_font_key, 'heading' );
	$body_font        = ravenpine_get_font_stack( $body_font_key, 'body' );

	$background_image_url = ravenpine_force_https_url( get_theme_mod( 'ravenpine_background_image', ravenpine_theme_mod_default( 'ravenpine_background_image' ) ) );
	$background_image     = $background_image_url ? "url('" . $background_image_url . "')" : 'none';
	$menu_section_bg_image_url = ravenpine_force_https_url( get_theme_mod( 'ravenpine_menu_section_bg_image', ravenpine_theme_mod_default( 'ravenpine_menu_section_bg_image' ) ) );
	$menu_section_bg_image     = $menu_section_bg_image_url ? "url('" . $menu_section_bg_image_url . "')" : 'none';
	$about_section_bg_image_url = ravenpine_force_https_url( get_theme_mod( 'ravenpine_about_section_bg_image', ravenpine_theme_mod_default( 'ravenpine_about_section_bg_image' ) ) );
	$about_section_bg_image     = $about_section_bg_image_url ? "url('" . $about_section_bg_image_url . "')" : 'none';
	$gallery_section_bg_image_url = ravenpine_force_https_url( get_theme_mod( 'ravenpine_gallery_section_bg_image', ravenpine_theme_mod_default( 'ravenpine_gallery_section_bg_image' ) ) );
	$gallery_section_bg_image     = $gallery_section_bg_image_url ? "url('" . $gallery_section_bg_image_url . "')" : 'none';
	$location_section_bg_image_url = ravenpine_force_https_url( get_theme_mod( 'ravenpine_location_section_bg_image', ravenpine_theme_mod_default( 'ravenpine_location_section_bg_image' ) ) );
	$location_section_bg_image     = $location_section_bg_image_url ? "url('" . $location_section_bg_image_url . "')" : 'none';
	$social_section_bg_image_url = ravenpine_force_https_url( get_theme_mod( 'ravenpine_social_section_bg_image', ravenpine_theme_mod_default( 'ravenpine_social_section_bg_image' ) ) );
	$social_section_bg_image     = $social_section_bg_image_url ? "url('" . $social_section_bg_image_url . "')" : 'none';
	$footer_bg_image_url = ravenpine_force_https_url( get_theme_mod( 'ravenpine_footer_bg_image', ravenpine_theme_mod_default( 'ravenpine_footer_bg_image' ) ) );
	$footer_bg_image     = $footer_bg_image_url ? "url('" . $footer_bg_image_url . "')" : 'none';

	$uploaded_font_choices = array();
	$font_face_css         = '';
	$selected_fonts        = array_unique( array( $heading_font_key, $body_font_key ) );

	foreach ( $selected_fonts as $font_key ) {
		if ( ravenpine_is_uploaded_font_key( $font_key ) ) {
			$uploaded_font_choices = ravenpine_get_uploaded_font_choices();
			break;
		}
	}

	foreach ( $selected_fonts as $font_key ) {
		if ( ! isset( $uploaded_font_choices[ $font_key ]['uploaded'] ) || empty( $uploaded_font_choices[ $font_key ]['uploaded'] ) ) {
			continue;
		}

		$font_face_css .= ravenpine_build_font_face_css(
			$uploaded_font_choices[ $font_key ]['url'] ?? '',
			$uploaded_font_choices[ $font_key ]['family'] ?? '',
			$uploaded_font_choices[ $font_key ]['style'] ?? 'normal',
			$uploaded_font_choices[ $font_key ]['weight'] ?? '400'
		);
	}

	$base_css = sprintf(
		':root{--rpx-color-accent:%1$s;--rpx-color-bg:%2$s;--rpx-color-fg:%3$s;--rpx-color-heading:%4$s;--rpx-surface:%5$s;--rpx-nav-bg:%6$s;--rpx-nav-link:%7$s;--rpx-hero-image-overlay:%8$s;--rpx-hero-card-bg:%9$s;--rpx-hero-card-border:%10$s;--rpx-hero-card-text:%11$s;--rpx-hero-button-bg:%12$s;--rpx-hero-button-text:%13$s;--rpx-menu-btn-bg:%14$s;--rpx-menu-btn-text:%15$s;--rpx-menu-btn-border:%16$s;--rpx-menu-section-bg:%17$s;--rpx-menu-section-bg-image:%18$s;--rpx-menu-section-heading:%19$s;--rpx-menu-section-text:%20$s;--rpx-location-card-bg:%21$s;--rpx-section-border-color:%22$s;--rpx-footer-bg:%23$s;--rpx-footer-bg-image:%24$s;--rpx-footer-text:%25$s;--rpx-tagline-color:%26$s;--rpx-section-padding:%27$spx;--rpx-section-border-top:%28$spx;--rpx-section-border-bottom:%29$spx;--rpx-font-heading:%30$s;--rpx-font-body:%31$s;--rpx-bg-image:%32$s;--rpx-about-section-bg:%33$s;--rpx-about-section-bg-image:%34$s;}',
		ravenpine_sanitize_css_value( $colors['ravenpine_accent_color'] ),
		ravenpine_sanitize_css_value( $colors['ravenpine_background_color'] ),
		ravenpine_sanitize_css_value( $colors['ravenpine_foreground_color'] ),
		ravenpine_sanitize_css_value( $colors['ravenpine_heading_color'] ),
		ravenpine_sanitize_css_value( $colors['ravenpine_surface_color'] ),
		ravenpine_sanitize_css_value( $colors['ravenpine_nav_bg_color'] ),
		ravenpine_sanitize_css_value( $colors['ravenpine_nav_link_color'] ),
		ravenpine_sanitize_css_value( $colors['ravenpine_hero_image_overlay_color'] ),
		ravenpine_sanitize_css_value( $colors['ravenpine_hero_card_bg_color'] ),
		ravenpine_sanitize_css_value( $colors['ravenpine_hero_card_border_color'] ),
		ravenpine_sanitize_css_value( $colors['ravenpine_hero_card_text_color'] ),
		ravenpine_sanitize_css_value( $colors['ravenpine_hero_button_bg_color'] ),
		ravenpine_sanitize_css_value( $colors['ravenpine_hero_button_text_color'] ),
		ravenpine_sanitize_css_value( $colors['ravenpine_menu_button_bg_color'] ),
		ravenpine_sanitize_css_value( $colors['ravenpine_menu_button_text_color'] ),
		ravenpine_sanitize_css_value( $colors['ravenpine_menu_button_border_color'] ),
		ravenpine_sanitize_css_value( $colors['ravenpine_menu_section_bg_color'] ),
		ravenpine_sanitize_css_value( $menu_section_bg_image ),
		ravenpine_sanitize_css_value( $colors['ravenpine_menu_section_heading_color'] ),
		ravenpine_sanitize_css_value( $colors['ravenpine_menu_section_text_color'] ),
		ravenpine_sanitize_css_value( $colors['ravenpine_location_card_bg_color'] ),
		ravenpine_sanitize_css_value( $colors['ravenpine_section_border_color'] ),
		ravenpine_sanitize_css_value( $colors['ravenpine_footer_bg_color'] ),
		ravenpine_sanitize_css_value( $footer_bg_image ),
		ravenpine_sanitize_css_value( $colors['ravenpine_footer_text_color'] ),
		ravenpine_sanitize_css_value( $colors['ravenpine_tagline_color'] ),
		(int) $pad,
		(int) $border_top_width,
		(int) $border_bottom_width,
		ravenpine_sanitize_css_value( $heading_font ),
		ravenpine_sanitize_css_value( $body_font ),
		ravenpine_sanitize_css_value( $background_image ),
		ravenpine_sanitize_css_value( $colors['ravenpine_about_section_bg_color'] ),
		ravenpine_sanitize_css_value( $about_section_bg_image )
	);

	$extra_block_css = sprintf(
		':root{--rpx-gallery-section-bg:%1$s;--rpx-gallery-section-bg-image:%2$s;--rpx-gallery-section-text:%3$s;--rpx-location-section-bg:%4$s;--rpx-location-section-bg-image:%5$s;--rpx-social-section-bg:%6$s;--rpx-social-section-bg-image:%7$s;--rpx-about-section-heading:%8$s;--rpx-about-section-text:%9$s;}',
		ravenpine_sanitize_css_value( $colors['ravenpine_gallery_section_bg_color'] ),
		ravenpine_sanitize_css_value( $gallery_section_bg_image ),
		ravenpine_sanitize_css_value( $colors['ravenpine_gallery_section_text_color'] ),
		ravenpine_sanitize_css_value( $colors['ravenpine_location_section_bg_color'] ),
		ravenpine_sanitize_css_value( $location_section_bg_image ),
		ravenpine_sanitize_css_value( $colors['ravenpine_social_section_bg_color'] ),
		ravenpine_sanitize_css_value( $social_section_bg_image ),
		ravenpine_sanitize_css_value( $colors['ravenpine_about_section_heading_color'] ),
		ravenpine_sanitize_css_value( $colors['ravenpine_about_section_text_color'] )
	);

	$css = $font_face_css . $base_css . $extra_block_css;

	if ( ravenpine_should_force_https_assets() ) {
		$css = str_replace( 'http://', 'https://', $css );
	}

	return $css;
}
