<?php
/**
 * Gallery manager settings screen.
 *
 * @package Ravenpine
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Parse and sanitize gallery image IDs.
 *
 * @param mixed $input Setting value.
 * @return array
 */
function ravenpine_sanitize_gallery_image_ids( $input ) {
	if ( is_string( $input ) ) {
		$input = explode( ',', $input );
	}

	if ( ! is_array( $input ) ) {
		return array();
	}

	$clean = array();
	foreach ( $input as $item ) {
		$id = absint( $item );
		if ( $id <= 0 ) {
			continue;
		}

		if ( 'attachment' !== get_post_type( $id ) ) {
			continue;
		}

		$mime = get_post_mime_type( $id );
		if ( ! is_string( $mime ) || 0 !== strpos( $mime, 'image/' ) ) {
			continue;
		}

		if ( ! in_array( $id, $clean, true ) ) {
			$clean[] = $id;
		}
	}

	return $clean;
}

/**
 * Register gallery manager setting.
 */
function ravenpine_register_gallery_settings() {
	register_setting(
		'ravenpine_gallery',
		'ravenpine_gallery_image_ids',
		array(
			'type'              => 'array',
			'sanitize_callback' => 'ravenpine_sanitize_gallery_image_ids',
			'default'           => array(),
		)
	);
}
add_action( 'admin_init', 'ravenpine_register_gallery_settings' );

/**
 * Get sanitized gallery image IDs.
 *
 * @return array
 */
function ravenpine_get_gallery_image_ids() {
	static $cached_ids = null;
	if ( null !== $cached_ids ) {
		return $cached_ids;
	}

	$raw = get_option( 'ravenpine_gallery_image_ids', array() );
	$ids = ravenpine_sanitize_gallery_image_ids( $raw );
	if ( ! empty( $ids ) ) {
		$cached_ids = $ids;
		return $cached_ids;
	}

	// Backward-compatible fallback from legacy per-post gallery entries.
	$legacy_posts = get_posts(
		array(
			'post_type'      => 'rpx_gallery',
			'post_status'    => 'publish',
			'posts_per_page' => -1,
			'orderby'        => array(
				'menu_order' => 'ASC',
				'date'       => 'DESC',
			),
			'fields'         => 'ids',
			'no_found_rows'  => true,
		)
	);

	if ( empty( $legacy_posts ) ) {
		$cached_ids = array();
		return $cached_ids;
	}

	$fallback_ids = array();
	foreach ( $legacy_posts as $legacy_id ) {
		$image_id = get_post_thumbnail_id( $legacy_id );
		if ( $image_id && ! in_array( $image_id, $fallback_ids, true ) ) {
			$fallback_ids[] = $image_id;
		}
	}

	$cached_ids = ravenpine_sanitize_gallery_image_ids( $fallback_ids );
	return $cached_ids;
}

/**
 * Add gallery manager page under Appearance.
 */
function ravenpine_add_gallery_manager_page() {
	add_theme_page(
		__( 'Gallery Manager', 'ravenpine' ),
		__( 'Gallery Manager', 'ravenpine' ),
		'manage_options',
		'ravenpine-gallery-manager',
		'ravenpine_render_gallery_manager_page'
	);
}
add_action( 'admin_menu', 'ravenpine_add_gallery_manager_page' );

/**
 * Render gallery manager screen.
 */
function ravenpine_render_gallery_manager_page() {
	if ( ! current_user_can( 'manage_options' ) ) {
		return;
	}

	$image_ids = ravenpine_get_gallery_image_ids();
	?>
	<div class="wrap">
		<h1><?php esc_html_e( 'Gallery Manager', 'ravenpine' ); ?></h1>
		<p><?php esc_html_e( 'Add many images at once, drag to reorder, and remove from one screen.', 'ravenpine' ); ?></p>
		<form method="post" action="options.php">
			<?php settings_fields( 'ravenpine_gallery' ); ?>
			<input type="hidden" id="ravenpine_gallery_image_ids" name="ravenpine_gallery_image_ids" value="<?php echo esc_attr( implode( ',', $image_ids ) ); ?>" />

			<div class="rpx-gallery-manager-actions">
				<button type="button" class="button button-primary" id="ravenpine-gallery-add-images"><?php esc_html_e( 'Add / Select Images', 'ravenpine' ); ?></button>
				<button type="button" class="button" id="ravenpine-gallery-clear-images"><?php esc_html_e( 'Clear All', 'ravenpine' ); ?></button>
			</div>

			<ul id="ravenpine-gallery-sortable" class="rpx-gallery-sortable">
				<?php foreach ( $image_ids as $image_id ) : ?>
					<?php
					$thumb = wp_get_attachment_image_url( $image_id, 'thumbnail' );
					$full  = wp_get_attachment_image_url( $image_id, 'full' );
					$name  = get_the_title( $image_id );
					if ( empty( $thumb ) ) {
						continue;
					}
					?>
					<li class="rpx-gallery-item" data-id="<?php echo esc_attr( (string) $image_id ); ?>">
						<div class="rpx-gallery-item-handle" aria-hidden="true">&#9776;</div>
						<img src="<?php echo esc_url( $thumb ); ?>" alt="" />
						<div class="rpx-gallery-item-meta">
							<strong><?php echo esc_html( $name ); ?></strong>
							<?php if ( $full ) : ?>
								<a href="<?php echo esc_url( $full ); ?>" target="_blank" rel="noopener noreferrer"><?php esc_html_e( 'Preview', 'ravenpine' ); ?></a>
							<?php endif; ?>
						</div>
						<button type="button" class="button-link-delete rpx-gallery-remove"><?php esc_html_e( 'Remove', 'ravenpine' ); ?></button>
					</li>
				<?php endforeach; ?>
			</ul>

			<p class="description"><?php esc_html_e( 'Tip: drag rows by the handle to reorder. The gallery block uses this exact order.', 'ravenpine' ); ?></p>

			<?php submit_button( __( 'Save Gallery', 'ravenpine' ) ); ?>
		</form>
	</div>
	<?php
}
