<?php
/**
 * Register admin content types and metadata.
 *
 * @package Ravenpine
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Register slider and menu post types.
 */
function ravenpine_register_post_types() {
	register_post_type(
		'rpx_slide',
		array(
			'labels' => array(
				'name'          => __( 'Hero Slides', 'ravenpine' ),
				'singular_name' => __( 'Hero Slide', 'ravenpine' ),
				'add_new_item'  => __( 'Add New Hero Slide', 'ravenpine' ),
				'edit_item'     => __( 'Edit Hero Slide', 'ravenpine' ),
			),
			'public'              => false,
			'show_ui'             => true,
			'show_in_rest'        => true,
			'menu_icon'           => 'dashicons-format-gallery',
			'menu_position'       => 20,
			'supports'            => array( 'title', 'editor', 'thumbnail', 'excerpt', 'page-attributes' ),
			'exclude_from_search' => true,
		)
	);

	register_post_type(
		'rpx_menu',
		array(
			'labels' => array(
				'name'          => __( 'Restaurant Menus', 'ravenpine' ),
				'singular_name' => __( 'Restaurant Menu', 'ravenpine' ),
				'add_new_item'  => __( 'Add New Restaurant Menu', 'ravenpine' ),
				'edit_item'     => __( 'Edit Restaurant Menu', 'ravenpine' ),
			),
			'public'              => false,
			'show_ui'             => true,
			'show_in_rest'        => true,
			'menu_icon'           => 'dashicons-media-document',
			'menu_position'       => 21,
			'supports'            => array( 'title', 'page-attributes' ),
			'exclude_from_search' => true,
		)
	);

	register_post_type(
		'rpx_gallery',
		array(
			'labels' => array(
				'name'          => __( 'Gallery Images', 'ravenpine' ),
				'singular_name' => __( 'Gallery Image', 'ravenpine' ),
				'add_new_item'  => __( 'Add New Gallery Image', 'ravenpine' ),
				'edit_item'     => __( 'Edit Gallery Image', 'ravenpine' ),
			),
			'public'              => false,
			'show_ui'             => false,
			'show_in_rest'        => false,
			'show_in_menu'        => false,
			'menu_icon'           => 'dashicons-format-image',
			'menu_position'       => 22,
			'supports'            => array( 'title', 'thumbnail', 'page-attributes' ),
			'exclude_from_search' => true,
		)
	);
}
add_action( 'init', 'ravenpine_register_post_types' );

/**
 * Register menu file URL meta box.
 */
function ravenpine_register_menu_meta_box() {
	add_meta_box(
		'ravenpine-menu-file',
		__( 'Menu File / Link', 'ravenpine' ),
		'ravenpine_render_menu_meta_box',
		'rpx_menu',
		'normal',
		'high'
	);
}
add_action( 'add_meta_boxes_rpx_menu', 'ravenpine_register_menu_meta_box' );

/**
 * Render meta box for menu link URL.
 *
 * @param WP_Post $post Current post object.
 */
function ravenpine_render_menu_meta_box( $post ) {
	wp_nonce_field( 'ravenpine_menu_meta', 'ravenpine_menu_meta_nonce' );
	$current_url = get_post_meta( $post->ID, '_ravenpine_menu_url', true );
	?>
	<p>
		<label for="ravenpine_menu_url"><strong><?php esc_html_e( 'Menu destination URL', 'ravenpine' ); ?></strong></label>
	</p>
	<p>
		<input type="url" class="widefat" id="ravenpine_menu_url" name="ravenpine_menu_url" value="<?php echo esc_attr( $current_url ); ?>" placeholder="https://example.com/menu.pdf" />
	</p>
	<p>
		<button type="button" class="button" id="ravenpine-upload-menu-file"><?php esc_html_e( 'Select / Upload PDF or Image', 'ravenpine' ); ?></button>
	</p>
	<p class="description">
		<?php esc_html_e( 'Each menu item appears as a button on the homepage block and opens this URL.', 'ravenpine' ); ?>
	</p>
	<?php
}

/**
 * Save menu link metadata.
 *
 * @param int $post_id Post ID.
 */
function ravenpine_save_menu_meta( $post_id ) {
	if ( ! isset( $_POST['ravenpine_menu_meta_nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['ravenpine_menu_meta_nonce'] ) ), 'ravenpine_menu_meta' ) ) {
		return;
	}

	if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
		return;
	}

	if ( ! current_user_can( 'edit_post', $post_id ) ) {
		return;
	}

	if ( isset( $_POST['ravenpine_menu_url'] ) ) {
		update_post_meta( $post_id, '_ravenpine_menu_url', esc_url_raw( wp_unslash( $_POST['ravenpine_menu_url'] ) ) );
	}
}
add_action( 'save_post_rpx_menu', 'ravenpine_save_menu_meta' );

/**
 * Add URL column to menu list screen.
 *
 * @param array $columns Existing columns.
 * @return array
 */
function ravenpine_menu_columns( $columns ) {
	$columns['menu_url'] = __( 'Menu URL', 'ravenpine' );
	return $columns;
}
add_filter( 'manage_rpx_menu_posts_columns', 'ravenpine_menu_columns' );

/**
 * Populate URL column.
 *
 * @param string $column  Column key.
 * @param int    $post_id Post ID.
 */
function ravenpine_menu_columns_content( $column, $post_id ) {
	if ( 'menu_url' !== $column ) {
		return;
	}

	$url = get_post_meta( $post_id, '_ravenpine_menu_url', true );
	if ( empty( $url ) ) {
		echo '<em>' . esc_html__( 'Not set', 'ravenpine' ) . '</em>';
		return;
	}

	echo '<a href="' . esc_url( $url ) . '" target="_blank" rel="noopener noreferrer">' . esc_html__( 'Open', 'ravenpine' ) . '</a>';
}
add_action( 'manage_rpx_menu_posts_custom_column', 'ravenpine_menu_columns_content', 10, 2 );
