<?php
/**
 * Restaurant settings page.
 *
 * @package Ravenpine
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Register option for location and hours.
 */
function ravenpine_register_restaurant_settings() {
	register_setting(
		'ravenpine_restaurant',
		'ravenpine_restaurant_settings',
		array(
			'type'              => 'array',
			'sanitize_callback' => 'ravenpine_sanitize_restaurant_settings',
			'default'           => ravenpine_get_restaurant_settings_defaults(),
		)
	);
}
add_action( 'admin_init', 'ravenpine_register_restaurant_settings' );

/**
 * Add settings page under Appearance.
 */
function ravenpine_add_restaurant_settings_page() {
	add_theme_page(
		__( 'Restaurant Info', 'ravenpine' ),
		__( 'Restaurant Info', 'ravenpine' ),
		'manage_options',
		'ravenpine-restaurant-settings',
		'ravenpine_render_restaurant_settings_page'
	);
}
add_action( 'admin_menu', 'ravenpine_add_restaurant_settings_page' );

/**
 * Defaults for settings.
 *
 * @return array
 */
function ravenpine_get_restaurant_settings_defaults() {
	return array(
		'address'                     => '',
		'phone'                       => '',
		'email'                       => '',
		'map_embed_url'               => '',
		'about_us_heading'            => 'About Us',
		'about_us_body'               => '',
		'about_us_image_url'          => '',
		'breakfast_hours'             => '7:30am - 11:00am, Seven Days a Week',
		'lunch_dinner_hours_line_one' => '11:00am - 10:00pm, Sunday - Thursday',
		'lunch_dinner_hours_line_two' => '11:00am - 11:00pm, Friday & Saturday',
		'happy_hour_line_one'         => '2:00pm - 5:00pm, Seven Days a Week',
		'happy_hour_line_two'         => '9:00pm - Close, Seven Days a Week',
		'additional_info'             => '',
		'popup_enabled'               => 0,
		'popup_width_px'              => 640,
		'popup_max_height_vh'         => 82,
		'popup_delay_seconds'         => 2,
		'popup_frequency_hours'       => 24,
		'popup_start_date'            => '',
		'popup_end_date'              => '',
		'popup_content'               => '',
	);
}

/**
 * Get settings with defaults.
 *
 * @return array
 */
function ravenpine_get_restaurant_settings() {
	$current = get_option( 'ravenpine_restaurant_settings', array() );
	if ( ! is_array( $current ) ) {
		$current = array();
	}

	return wp_parse_args( $current, ravenpine_get_restaurant_settings_defaults() );
}

/**
 * Sanitize settings payload.
 *
 * @param array $input Raw input.
 * @return array
 */
function ravenpine_sanitize_restaurant_settings( $input ) {
	$input = is_array( $input ) ? $input : array();

	$output = array();
	$output['address']       = isset( $input['address'] ) ? sanitize_textarea_field( $input['address'] ) : '';
	$output['phone']         = isset( $input['phone'] ) ? sanitize_text_field( $input['phone'] ) : '';
	$output['email']         = isset( $input['email'] ) ? sanitize_email( $input['email'] ) : '';
	$output['map_embed_url'] = isset( $input['map_embed_url'] ) ? esc_url_raw( $input['map_embed_url'] ) : '';
	$output['about_us_heading']   = isset( $input['about_us_heading'] ) ? sanitize_text_field( $input['about_us_heading'] ) : '';
	$output['about_us_body']      = isset( $input['about_us_body'] ) ? sanitize_textarea_field( $input['about_us_body'] ) : '';
	$output['about_us_image_url'] = isset( $input['about_us_image_url'] ) ? esc_url_raw( $input['about_us_image_url'] ) : '';

	$hour_fields = array(
		'breakfast_hours',
		'lunch_dinner_hours_line_one',
		'lunch_dinner_hours_line_two',
		'happy_hour_line_one',
		'happy_hour_line_two',
	);

	foreach ( $hour_fields as $field ) {
		$output[ $field ] = isset( $input[ $field ] ) ? sanitize_text_field( $input[ $field ] ) : '';
	}

	$output['additional_info']       = isset( $input['additional_info'] ) ? wp_kses_post( (string) $input['additional_info'] ) : '';
	$output['popup_enabled']         = ! empty( $input['popup_enabled'] ) ? 1 : 0;
	$output['popup_width_px']        = isset( $input['popup_width_px'] ) ? absint( $input['popup_width_px'] ) : 640;
	$output['popup_max_height_vh']   = isset( $input['popup_max_height_vh'] ) ? absint( $input['popup_max_height_vh'] ) : 82;
	$output['popup_delay_seconds']   = isset( $input['popup_delay_seconds'] ) ? absint( $input['popup_delay_seconds'] ) : 2;
	$output['popup_frequency_hours'] = isset( $input['popup_frequency_hours'] ) ? absint( $input['popup_frequency_hours'] ) : 24;
	$output['popup_start_date']      = isset( $input['popup_start_date'] ) ? sanitize_text_field( $input['popup_start_date'] ) : '';
	$output['popup_end_date']        = isset( $input['popup_end_date'] ) ? sanitize_text_field( $input['popup_end_date'] ) : '';
	$output['popup_content']         = isset( $input['popup_content'] ) ? wp_kses_post( (string) $input['popup_content'] ) : '';

	$output['popup_width_px']      = min( max( $output['popup_width_px'], 280 ), 1200 );
	$output['popup_max_height_vh'] = min( max( $output['popup_max_height_vh'], 40 ), 96 );
	$output['popup_delay_seconds'] = min( max( $output['popup_delay_seconds'], 0 ), 120 );
	$output['popup_frequency_hours'] = min( max( $output['popup_frequency_hours'], 0 ), 8760 );

	if ( ! preg_match( '/^\d{4}-\d{2}-\d{2}$/', $output['popup_start_date'] ) ) {
		$output['popup_start_date'] = '';
	}
	if ( ! preg_match( '/^\d{4}-\d{2}-\d{2}$/', $output['popup_end_date'] ) ) {
		$output['popup_end_date'] = '';
	}

	return $output;
}

/**
 * Render settings admin page.
 */
function ravenpine_render_restaurant_settings_page() {
	if ( ! current_user_can( 'manage_options' ) ) {
		return;
	}

	$settings = ravenpine_get_restaurant_settings();
	$timezone_label = wp_timezone_string();
	if ( empty( $timezone_label ) ) {
		$timezone_label = 'UTC';
	}
	?>
	<div class="wrap">
		<h1><?php esc_html_e( 'Restaurant Info', 'ravenpine' ); ?></h1>
		<p><?php esc_html_e( 'Manage About Us, location, contact, hours, and map settings used by theme blocks.', 'ravenpine' ); ?></p>
		<form method="post" action="options.php">
			<?php settings_fields( 'ravenpine_restaurant' ); ?>
			<table class="form-table" role="presentation">
				<tr>
					<th scope="row"><label for="ravenpine_address"><?php esc_html_e( 'Address', 'ravenpine' ); ?></label></th>
					<td>
						<textarea id="ravenpine_address" name="ravenpine_restaurant_settings[address]" rows="3" class="large-text"><?php echo esc_textarea( $settings['address'] ); ?></textarea>
					</td>
				</tr>
				<tr>
					<th scope="row"><label for="ravenpine_phone"><?php esc_html_e( 'Phone', 'ravenpine' ); ?></label></th>
					<td>
						<input id="ravenpine_phone" name="ravenpine_restaurant_settings[phone]" type="text" class="regular-text" value="<?php echo esc_attr( $settings['phone'] ); ?>" />
					</td>
				</tr>
				<tr>
					<th scope="row"><label for="ravenpine_email"><?php esc_html_e( 'Email', 'ravenpine' ); ?></label></th>
					<td>
						<input id="ravenpine_email" name="ravenpine_restaurant_settings[email]" type="email" class="regular-text" value="<?php echo esc_attr( $settings['email'] ); ?>" />
					</td>
				</tr>
				<tr>
					<th scope="row"><label for="ravenpine_map_embed_url"><?php esc_html_e( 'Google Map Embed URL', 'ravenpine' ); ?></label></th>
					<td>
						<input id="ravenpine_map_embed_url" name="ravenpine_restaurant_settings[map_embed_url]" type="url" class="large-text" value="<?php echo esc_attr( $settings['map_embed_url'] ); ?>" />
						<p class="description"><?php esc_html_e( 'Use a full Google Maps embed URL, or leave blank to auto-generate from the address.', 'ravenpine' ); ?></p>
					</td>
				</tr>
				<tr>
					<th scope="row"><?php esc_html_e( 'About Us', 'ravenpine' ); ?></th>
					<td>
						<fieldset>
							<label for="ravenpine_about_us_heading"><strong><?php esc_html_e( 'Heading', 'ravenpine' ); ?></strong></label><br />
							<input id="ravenpine_about_us_heading" name="ravenpine_restaurant_settings[about_us_heading]" type="text" class="large-text" value="<?php echo esc_attr( $settings['about_us_heading'] ); ?>" /><br /><br />
							<label for="ravenpine_about_us_body"><strong><?php esc_html_e( 'Body Text', 'ravenpine' ); ?></strong></label><br />
							<textarea id="ravenpine_about_us_body" name="ravenpine_restaurant_settings[about_us_body]" rows="6" class="large-text"><?php echo esc_textarea( $settings['about_us_body'] ); ?></textarea>
							<p class="description"><?php esc_html_e( 'Use line breaks to create separate paragraphs.', 'ravenpine' ); ?></p>
							<label for="ravenpine_about_us_image_url"><strong><?php esc_html_e( 'About Us Image', 'ravenpine' ); ?></strong></label><br />
							<div class="rpx-admin-image-picker">
								<input id="ravenpine_about_us_image_url" name="ravenpine_restaurant_settings[about_us_image_url]" type="url" class="large-text rpx-media-url" value="<?php echo esc_attr( $settings['about_us_image_url'] ); ?>" readonly />
								<p class="description"><?php esc_html_e( 'Select an image from the Media Library. If no image is selected, About Us text is centered.', 'ravenpine' ); ?></p>
								<p class="rpx-media-actions">
									<button type="button" class="button rpx-media-upload"><?php esc_html_e( 'Upload / Select Image', 'ravenpine' ); ?></button>
									<button type="button" class="button-link-delete rpx-media-remove<?php echo empty( $settings['about_us_image_url'] ) ? ' is-hidden' : ''; ?>"><?php esc_html_e( 'Remove Image', 'ravenpine' ); ?></button>
								</p>
								<div class="rpx-media-preview<?php echo empty( $settings['about_us_image_url'] ) ? ' is-hidden' : ''; ?>">
									<img src="<?php echo esc_url( $settings['about_us_image_url'] ); ?>" alt="" />
								</div>
							</div>
						</fieldset>
					</td>
				</tr>
				<tr>
					<th scope="row"><?php esc_html_e( 'Hours of Operation', 'ravenpine' ); ?></th>
					<td>
						<fieldset>
							<label for="ravenpine_breakfast_hours"><strong><?php esc_html_e( 'Breakfast', 'ravenpine' ); ?></strong></label><br />
							<input id="ravenpine_breakfast_hours" name="ravenpine_restaurant_settings[breakfast_hours]" type="text" class="large-text" value="<?php echo esc_attr( $settings['breakfast_hours'] ); ?>" /><br /><br />
							<label for="ravenpine_lunch_dinner_hours_line_one"><strong><?php esc_html_e( 'Lunch & Dinner (Line 1)', 'ravenpine' ); ?></strong></label><br />
							<input id="ravenpine_lunch_dinner_hours_line_one" name="ravenpine_restaurant_settings[lunch_dinner_hours_line_one]" type="text" class="large-text" value="<?php echo esc_attr( $settings['lunch_dinner_hours_line_one'] ); ?>" /><br /><br />
							<label for="ravenpine_lunch_dinner_hours_line_two"><strong><?php esc_html_e( 'Lunch & Dinner (Line 2)', 'ravenpine' ); ?></strong></label><br />
							<input id="ravenpine_lunch_dinner_hours_line_two" name="ravenpine_restaurant_settings[lunch_dinner_hours_line_two]" type="text" class="large-text" value="<?php echo esc_attr( $settings['lunch_dinner_hours_line_two'] ); ?>" /><br /><br />
							<label for="ravenpine_happy_hour_line_one"><strong><?php esc_html_e( 'Happy Hour (Line 1)', 'ravenpine' ); ?></strong></label><br />
							<input id="ravenpine_happy_hour_line_one" name="ravenpine_restaurant_settings[happy_hour_line_one]" type="text" class="large-text" value="<?php echo esc_attr( $settings['happy_hour_line_one'] ); ?>" /><br /><br />
							<label for="ravenpine_happy_hour_line_two"><strong><?php esc_html_e( 'Happy Hour (Line 2)', 'ravenpine' ); ?></strong></label><br />
							<input id="ravenpine_happy_hour_line_two" name="ravenpine_restaurant_settings[happy_hour_line_two]" type="text" class="large-text" value="<?php echo esc_attr( $settings['happy_hour_line_two'] ); ?>" />
						</fieldset>
					</td>
				</tr>
				<tr>
					<th scope="row"><?php esc_html_e( 'Footer Additional Info', 'ravenpine' ); ?></th>
					<td>
						<fieldset>
							<label for="ravenpine_additional_info"><strong><?php esc_html_e( 'Additional Info', 'ravenpine' ); ?></strong></label>
							<div class="rpx-editor-theme-toggle">
								<button
									type="button"
									class="button"
									data-rpx-editor-dark-toggle="ravenpine_additional_info"
									aria-pressed="false"
								>
									<?php esc_html_e( 'Enable Editor Dark Mode', 'ravenpine' ); ?>
								</button>
								<p class="description"><?php esc_html_e( 'Editor-only preview for easier light-text editing.', 'ravenpine' ); ?></p>
							</div>
							<?php
							wp_editor(
								$settings['additional_info'],
								'ravenpine_additional_info',
								array(
									'textarea_name' => 'ravenpine_restaurant_settings[additional_info]',
									'textarea_rows' => 8,
									'media_buttons' => true,
									'teeny'         => false,
								)
							);
							?>
							<p class="description"><?php esc_html_e( 'Optional middle-column content shown in the site footer.', 'ravenpine' ); ?></p>
						</fieldset>
					</td>
				</tr>
				<tr>
					<th scope="row"><?php esc_html_e( 'Popup Modal', 'ravenpine' ); ?></th>
					<td>
						<fieldset class="rpx-popup-settings">
							<label for="ravenpine_popup_enabled">
								<input id="ravenpine_popup_enabled" name="ravenpine_restaurant_settings[popup_enabled]" type="checkbox" value="1" <?php checked( ! empty( $settings['popup_enabled'] ) ); ?> />
								<strong><?php esc_html_e( 'Enable popup modal', 'ravenpine' ); ?></strong>
							</label>

							<div class="rpx-popup-settings-grid">
								<div>
									<label for="ravenpine_popup_width_px"><strong><?php esc_html_e( 'Popup Width (px)', 'ravenpine' ); ?></strong></label><br />
									<input id="ravenpine_popup_width_px" name="ravenpine_restaurant_settings[popup_width_px]" type="number" min="280" max="1200" step="10" class="small-text" value="<?php echo esc_attr( $settings['popup_width_px'] ); ?>" />
								</div>
								<div>
									<label for="ravenpine_popup_max_height_vh"><strong><?php esc_html_e( 'Popup Max Height (vh)', 'ravenpine' ); ?></strong></label><br />
									<input id="ravenpine_popup_max_height_vh" name="ravenpine_restaurant_settings[popup_max_height_vh]" type="number" min="40" max="96" step="1" class="small-text" value="<?php echo esc_attr( $settings['popup_max_height_vh'] ); ?>" />
								</div>
								<div>
									<label for="ravenpine_popup_delay_seconds"><strong><?php esc_html_e( 'Delay Before Popup (seconds)', 'ravenpine' ); ?></strong></label><br />
									<input id="ravenpine_popup_delay_seconds" name="ravenpine_restaurant_settings[popup_delay_seconds]" type="number" min="0" max="120" step="1" class="small-text" value="<?php echo esc_attr( $settings['popup_delay_seconds'] ); ?>" />
								</div>
								<div>
									<label for="ravenpine_popup_frequency_hours"><strong><?php esc_html_e( 'Show Again After (hours)', 'ravenpine' ); ?></strong></label><br />
									<input id="ravenpine_popup_frequency_hours" name="ravenpine_restaurant_settings[popup_frequency_hours]" type="number" min="0" max="8760" step="1" class="small-text" value="<?php echo esc_attr( $settings['popup_frequency_hours'] ); ?>" />
									<p class="description"><?php esc_html_e( 'Use 0 to show every page load.', 'ravenpine' ); ?></p>
								</div>
							</div>

							<div class="rpx-popup-settings-grid">
								<div>
									<label for="ravenpine_popup_start_date"><strong><?php esc_html_e( 'Start Date', 'ravenpine' ); ?></strong></label><br />
									<input id="ravenpine_popup_start_date" name="ravenpine_restaurant_settings[popup_start_date]" type="date" value="<?php echo esc_attr( $settings['popup_start_date'] ); ?>" />
								</div>
								<div>
									<label for="ravenpine_popup_end_date"><strong><?php esc_html_e( 'End Date', 'ravenpine' ); ?></strong></label><br />
									<input id="ravenpine_popup_end_date" name="ravenpine_restaurant_settings[popup_end_date]" type="date" value="<?php echo esc_attr( $settings['popup_end_date'] ); ?>" />
								</div>
							</div>
							<p class="description"><?php echo esc_html( sprintf( __( 'Date range uses site timezone: %s', 'ravenpine' ), $timezone_label ) ); ?></p>

							<label for="ravenpine_popup_content"><strong><?php esc_html_e( 'Popup Content', 'ravenpine' ); ?></strong></label>
							<?php
							wp_editor(
								$settings['popup_content'],
								'ravenpine_popup_content',
								array(
									'textarea_name' => 'ravenpine_restaurant_settings[popup_content]',
									'textarea_rows' => 10,
									'media_buttons' => true,
									'teeny'         => false,
								)
							);
							?>
						</fieldset>
					</td>
				</tr>
			</table>
			<?php submit_button(); ?>
		</form>
	</div>
	<?php
}

/**
 * Convert popup schedule date to timestamp in site timezone.
 *
 * @param string $date       Date (Y-m-d).
 * @param bool   $end_of_day Whether to use end-of-day timestamp.
 * @return int|null
 */
function ravenpine_popup_date_to_timestamp( $date, $end_of_day = false ) {
	$date = trim( (string) $date );
	if ( ! preg_match( '/^\d{4}-\d{2}-\d{2}$/', $date ) ) {
		return null;
	}

	$time = $end_of_day ? '23:59:59' : '00:00:00';
	$dt   = date_create_immutable_from_format( 'Y-m-d H:i:s', $date . ' ' . $time, wp_timezone() );
	if ( false === $dt ) {
		return null;
	}

	return $dt->getTimestamp();
}

/**
 * Check popup date-range activation.
 *
 * @param array $settings Restaurant settings.
 * @return bool
 */
function ravenpine_popup_is_in_active_date_range( $settings ) {
	$start_ts = ravenpine_popup_date_to_timestamp( $settings['popup_start_date'] ?? '', false );
	$end_ts   = ravenpine_popup_date_to_timestamp( $settings['popup_end_date'] ?? '', true );
	$now_ts   = current_datetime()->getTimestamp();

	if ( null !== $start_ts && null !== $end_ts && $start_ts > $end_ts ) {
		return false;
	}

	if ( null !== $start_ts && $now_ts < $start_ts ) {
		return false;
	}

	if ( null !== $end_ts && $now_ts > $end_ts ) {
		return false;
	}

	return true;
}

/**
 * Render promotional popup modal on front-end.
 */
function ravenpine_render_popup_modal() {
	if ( is_admin() ) {
		return;
	}

	$settings = ravenpine_get_restaurant_settings();
	if ( empty( $settings['popup_enabled'] ) ) {
		return;
	}

	$content = isset( $settings['popup_content'] ) ? trim( (string) $settings['popup_content'] ) : '';
	if ( '' === $content ) {
		return;
	}

	if ( ! ravenpine_popup_is_in_active_date_range( $settings ) ) {
		return;
	}

	$popup_width_px      = min( max( absint( $settings['popup_width_px'] ?? 640 ), 280 ), 1200 );
	$popup_max_height_vh = min( max( absint( $settings['popup_max_height_vh'] ?? 82 ), 40 ), 96 );
	$delay_ms            = min( max( absint( $settings['popup_delay_seconds'] ?? 2 ), 0 ), 120 ) * 1000;
	$frequency_hours     = min( max( absint( $settings['popup_frequency_hours'] ?? 24 ), 0 ), 8760 );
	$storage_hash_source = home_url( '/' ) . '|' . ( $settings['popup_start_date'] ?? '' ) . '|' . ( $settings['popup_end_date'] ?? '' ) . '|' . wp_strip_all_tags( $content );
	$storage_key         = 'rpx_popup_seen_' . md5( $storage_hash_source );
	$style               = '--rpx-popup-width:' . $popup_width_px . 'px;--rpx-popup-max-height:' . $popup_max_height_vh . 'vh;';
	?>
	<div
		class="rpx-popup"
		data-rpx-popup
		data-rpx-popup-delay="<?php echo esc_attr( (string) $delay_ms ); ?>"
		data-rpx-popup-frequency-hours="<?php echo esc_attr( (string) $frequency_hours ); ?>"
		data-rpx-popup-storage-key="<?php echo esc_attr( $storage_key ); ?>"
		hidden
	>
		<div class="rpx-popup__backdrop" data-rpx-popup-close></div>
		<div class="rpx-popup__dialog" role="dialog" aria-modal="true" aria-label="<?php esc_attr_e( 'Promotional popup', 'ravenpine' ); ?>" style="<?php echo esc_attr( $style ); ?>">
			<button class="rpx-popup__close" type="button" data-rpx-popup-close aria-label="<?php esc_attr_e( 'Close popup', 'ravenpine' ); ?>">&times;</button>
			<div class="rpx-popup__content">
				<?php echo wp_kses_post( $content ); ?>
			</div>
		</div>
	</div>
	<script>
		(function () {
			var root = document.querySelector('[data-rpx-popup]');
			if (!root || window.__rpxPopupInit) {
				return;
			}
			window.__rpxPopupInit = true;

			// Ensure popup is attached directly to body and rendered last so
			// parent stacking contexts cannot render it behind page content.
			if (document.body) {
				document.body.appendChild(root);
			}
			root.style.position = 'fixed';
			root.style.inset = '0';
			root.style.zIndex = '2147483647';

			var delayMs = parseInt(root.getAttribute('data-rpx-popup-delay') || '0', 10);
			var frequencyHours = parseInt(root.getAttribute('data-rpx-popup-frequency-hours') || '0', 10);
			var storageKey = root.getAttribute('data-rpx-popup-storage-key') || 'rpx_popup_seen';
			var closeButton = root.querySelector('.rpx-popup__close');

			if (isNaN(delayMs) || delayMs < 0) {
				delayMs = 0;
			}
			if (isNaN(frequencyHours) || frequencyHours < 0) {
				frequencyHours = 0;
			}

			function canShowPopup() {
				var search = String(window.location.search || '');
				if (search.indexOf('popup_preview=1') !== -1 || search.indexOf('no-cache=1') !== -1) {
					return true;
				}

				if (frequencyHours <= 0) {
					return true;
				}

				try {
					var stored = localStorage.getItem(storageKey);
					if (!stored) {
						return true;
					}

					var lastSeen = parseInt(stored, 10);
					if (isNaN(lastSeen)) {
						return true;
					}

					return (Date.now() - lastSeen) >= (frequencyHours * 60 * 60 * 1000);
				} catch (error) {
					return true;
				}
			}

			function markShown() {
				if (frequencyHours <= 0) {
					return;
				}

				try {
					localStorage.setItem(storageKey, String(Date.now()));
				} catch (error) {
					// Ignore storage failures.
				}
			}

			function openPopup() {
				root.hidden = false;
				root.removeAttribute('hidden');
				root.classList.add('is-active');
				document.body.classList.add('rpx-popup-open');

				if (closeButton) {
					try {
						closeButton.focus({ preventScroll: true });
					} catch (error) {
						closeButton.focus();
					}
				}

				markShown();
			}

			function closePopup() {
				root.classList.remove('is-active');
				document.body.classList.remove('rpx-popup-open');
				window.setTimeout(function () {
					if (!root.classList.contains('is-active')) {
						root.hidden = true;
						root.setAttribute('hidden', 'hidden');
					}
				}, 260);
			}

			root.addEventListener('click', function (event) {
				if (event.target && event.target.hasAttribute('data-rpx-popup-close')) {
					closePopup();
				}
			});

			document.addEventListener('keydown', function (event) {
				if (event.key === 'Escape' && root.classList.contains('is-active')) {
					closePopup();
				}
			});

			if (!canShowPopup()) {
				return;
			}

			window.setTimeout(openPopup, delayMs);
		})();
	</script>
	<?php
}
add_action( 'wp_footer', 'ravenpine_render_popup_modal', 60 );
