<?php
/**
 * Theme setup and asset loading.
 *
 * @package Ravenpine
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Register theme supports and defaults.
 */
function ravenpine_setup() {
	load_theme_textdomain( 'ravenpine', get_template_directory() . '/languages' );

	add_theme_support( 'title-tag' );
	add_theme_support( 'post-thumbnails' );
	add_theme_support( 'wp-block-styles' );
	add_theme_support( 'responsive-embeds' );
	add_theme_support( 'align-wide' );
	add_theme_support( 'editor-styles' );
	add_editor_style( 'assets/css/editor.css' );

	add_theme_support(
		'custom-logo',
		array(
			'height'      => 160,
			'width'       => 320,
			'flex-height' => true,
			'flex-width'  => true,
		)
	);

	register_block_pattern_category(
		'ravenpine',
		array(
			'label' => __( 'Ravenpine', 'ravenpine' ),
		)
	);
}
add_action( 'after_setup_theme', 'ravenpine_setup' );

/**
 * Resolve a cache-busting asset version from file modified time.
 *
 * @param string $relative_path Path relative to theme root.
 * @return string
 */
function ravenpine_asset_version( $relative_path ) {
	$relative_path = ltrim( (string) $relative_path, '/' );
	$file_path     = trailingslashit( get_template_directory() ) . $relative_path;
	if ( file_exists( $file_path ) ) {
		return (string) filemtime( $file_path );
	}

	return (string) wp_get_theme()->get( 'Version' );
}

/**
 * Enqueue front-end assets.
 */
function ravenpine_enqueue_assets() {
	$deps    = array( 'wp-block-library' );

	if ( function_exists( 'ravenpine_get_google_fonts_url' ) ) {
		$fonts_url = ravenpine_get_google_fonts_url();
		if ( ! empty( $fonts_url ) ) {
			wp_enqueue_style(
				'ravenpine-fonts',
				$fonts_url,
				array(),
				null
			);
			$deps[] = 'ravenpine-fonts';
		}
	}

	wp_enqueue_style(
		'ravenpine-theme',
		get_template_directory_uri() . '/assets/css/theme.css',
		$deps,
		ravenpine_asset_version( 'assets/css/theme.css' )
	);

	wp_register_script(
		'ravenpine-hero-slider',
		get_template_directory_uri() . '/assets/js/hero-slider.js',
		array(),
		ravenpine_asset_version( 'assets/js/hero-slider.js' ),
		true
	);

	wp_register_script(
		'ravenpine-scroll-reveal',
		get_template_directory_uri() . '/assets/js/scroll-reveal.js',
		array(),
		ravenpine_asset_version( 'assets/js/scroll-reveal.js' ),
		true
	);

	wp_register_script(
		'ravenpine-gallery-lightbox',
		get_template_directory_uri() . '/assets/js/gallery-lightbox.js',
		array(),
		ravenpine_asset_version( 'assets/js/gallery-lightbox.js' ),
		true
	);

	wp_register_script(
		'ravenpine-gallery-carousel',
		get_template_directory_uri() . '/assets/js/gallery-carousel.js',
		array(),
		ravenpine_asset_version( 'assets/js/gallery-carousel.js' ),
		true
	);

	// Always load the hero slider runtime on the front end.
	// The script is safe on pages without a slider and guarantees controls/autoplay work.
	wp_enqueue_script( 'ravenpine-hero-slider' );

	// Always load gallery lightbox runtime to ensure click handlers are available
	// even when render timing or optimization plugins delay script execution.
	wp_enqueue_script( 'ravenpine-gallery-lightbox' );

	// Safe to always load; binds only when carousel markup exists.
	wp_enqueue_script( 'ravenpine-gallery-carousel' );

	if ( function_exists( 'ravenpine_get_customizer_css_vars' ) ) {
		wp_add_inline_style( 'ravenpine-theme', ravenpine_get_customizer_css_vars() );
	}
}
add_action( 'wp_enqueue_scripts', 'ravenpine_enqueue_assets' );

/**
 * Enqueue registered runtime script for dynamic blocks.
 *
 * @param string $handle Script handle.
 * @return void
 */
function ravenpine_enqueue_runtime_script( $handle ) {
	if ( is_admin() ) {
		return;
	}

	if ( wp_script_is( $handle, 'registered' ) ) {
		wp_enqueue_script( $handle );
	}
}

/**
 * Enqueue admin assets for menu file picker.
 *
 * @param string $hook_suffix Current admin page.
 */
function ravenpine_admin_assets( $hook_suffix ) {
	$screen  = get_current_screen();

	if ( in_array( $hook_suffix, array( 'post.php', 'post-new.php' ), true ) && $screen && 'rpx_menu' === $screen->post_type ) {
		wp_enqueue_media();
		wp_enqueue_script(
			'ravenpine-admin-menu-file',
			get_template_directory_uri() . '/assets/js/admin-menu-file.js',
			array( 'jquery' ),
			ravenpine_asset_version( 'assets/js/admin-menu-file.js' ),
			true
		);
	}

	if ( 'appearance_page_ravenpine-gallery-manager' === $hook_suffix ) {
		wp_enqueue_media();
		wp_enqueue_script(
			'ravenpine-admin-gallery-manager',
			get_template_directory_uri() . '/assets/js/admin-gallery-manager.js',
			array( 'jquery', 'jquery-ui-sortable' ),
			ravenpine_asset_version( 'assets/js/admin-gallery-manager.js' ),
			true
		);
		wp_enqueue_style(
			'ravenpine-admin-gallery-manager',
			get_template_directory_uri() . '/assets/css/admin-gallery-manager.css',
			array(),
			ravenpine_asset_version( 'assets/css/admin-gallery-manager.css' )
		);
	}

	if ( 'appearance_page_ravenpine-restaurant-settings' === $hook_suffix ) {
		wp_enqueue_media();
		wp_enqueue_script(
			'ravenpine-admin-restaurant-info',
			get_template_directory_uri() . '/assets/js/admin-restaurant-info.js',
			array( 'jquery' ),
			ravenpine_asset_version( 'assets/js/admin-restaurant-info.js' ),
			true
		);
		wp_enqueue_style(
			'ravenpine-admin-restaurant-info',
			get_template_directory_uri() . '/assets/css/admin-restaurant-info.css',
			array(),
			ravenpine_asset_version( 'assets/css/admin-restaurant-info.css' )
		);
	}
}
add_action( 'admin_enqueue_scripts', 'ravenpine_admin_assets' );

/**
 * Allow common font uploads for custom typography.
 *
 * @param array $mimes Existing mime types.
 * @return array
 */
function ravenpine_allow_font_upload_mimes( $mimes ) {
	$mimes['woff']   = 'font/woff';
	$mimes['woff2']  = 'font/woff2';
	$mimes['ttf']    = 'application/x-font-ttf';
	$mimes['otf']    = 'application/x-font-opentype';
	$mimes['ttf|otf|woff|woff2'] = 'application/octet-stream';
	return $mimes;
}
add_filter( 'upload_mimes', 'ravenpine_allow_font_upload_mimes' );

/**
 * Normalize font mime validation across hosts for uploads.
 *
 * Some environments report font files as application/octet-stream which causes
 * WordPress to reject them even when extension is allowed.
 *
 * @param array        $data      File data array.
 * @param string       $file      Full path to file.
 * @param string       $filename  File name.
 * @param array|string $mimes     Allowed mimes.
 * @param string|false $real_mime Real mime type from fileinfo.
 * @return array
 */
function ravenpine_fix_font_filetype_check( $data, $file, $filename, $mimes, $real_mime ) {
	$ext = strtolower( pathinfo( $filename, PATHINFO_EXTENSION ) );

	$font_mimes = array(
		'ttf'   => 'font/ttf',
		'otf'   => 'font/otf',
		'woff'  => 'font/woff',
		'woff2' => 'font/woff2',
	);

	if ( ! isset( $font_mimes[ $ext ] ) ) {
		return $data;
	}

	if ( ! empty( $data['ext'] ) && ! empty( $data['type'] ) ) {
		return $data;
	}

	$data['ext']  = $ext;
	$data['type'] = $font_mimes[ $ext ];
	$data['proper_filename'] = $filename;

	return $data;
}
add_filter( 'wp_check_filetype_and_ext', 'ravenpine_fix_font_filetype_check', 10, 5 );

/**
 * Hide Appearance editor entries to keep configuration in this theme system.
 */
function ravenpine_disable_appearance_editor_menu() {
	remove_submenu_page( 'themes.php', 'site-editor.php' );
	remove_submenu_page( 'themes.php', 'gutenberg-edit-site' );
	remove_submenu_page( 'themes.php', 'theme-editor.php' );
}
add_action( 'admin_menu', 'ravenpine_disable_appearance_editor_menu', 999 );

/**
 * Print font-face styles with HTTPS-normalized URLs.
 *
 * WordPress prints local font faces in wp_head via wp_print_font_faces() using
 * merged theme.json/user font data. In migrated environments those URLs can be
 * persisted as HTTP and trigger mixed-content blocking on HTTPS pages.
 *
 * @return void
 */
function ravenpine_print_font_faces_https() {
	if ( ! function_exists( 'wp_print_font_faces' ) ) {
		return;
	}

	ob_start();
	wp_print_font_faces();
	$font_face_output = ob_get_clean();

	if ( function_exists( 'ravenpine_should_force_https_assets' ) && ravenpine_should_force_https_assets() ) {
		$font_face_output = str_replace( 'http://', 'https://', (string) $font_face_output );
	}

	echo $font_face_output; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
}

/**
 * Replace core font-face output with HTTPS-normalized output.
 *
 * @return void
 */
function ravenpine_override_wp_font_face_output() {
	remove_action( 'wp_head', 'wp_print_font_faces', 50 );
	add_action( 'wp_head', 'ravenpine_print_font_faces_https', 50 );
}
add_action( 'init', 'ravenpine_override_wp_font_face_output' );

/**
 * Block direct access to Site Editor screen.
 */
function ravenpine_block_site_editor_access() {
	global $pagenow;

	if ( 'site-editor.php' !== $pagenow ) {
		return;
	}

	wp_safe_redirect( admin_url( 'customize.php' ) );
	exit;
}
add_action( 'admin_init', 'ravenpine_block_site_editor_access' );
