<?php
/**
 * Plugin Name: WP Unconfirmed Users
 * Description: Shows pending invited users in the Users screen and lets admins resend, activate, or delete invitations.
 * Version: 1.0.0
 * Author: Lance Smith
 * License: GPL-2.0-or-later
 * Text Domain: wpunconfirmed
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'WP_Unconfirmed_Users_Plugin' ) ) {
	final class WP_Unconfirmed_Users_Plugin {
		const ROLE_FILTER = 'wpunconfirmed';

		const ACTION_RESEND   = 'wpunconfirmed_resend';
		const ACTION_ACTIVATE = 'wpunconfirmed_activate';
		const ACTION_DELETE   = 'wpunconfirmed_delete';

		/** @var WP_Unconfirmed_Users_Plugin|null */
		private static $instance = null;

		/** @var array<string,array<int,stdClass>> */
		private $pending_cache = array();

		/** @var array<int,string> */
		private $site_label_cache = array();

		/** @var bool|null */
		private $signups_table_exists = null;

		/**
		 * @return WP_Unconfirmed_Users_Plugin
		 */
		public static function instance() {
			if ( null === self::$instance ) {
				self::$instance = new self();
			}

			return self::$instance;
		}

		private function __construct() {
			if ( ! is_admin() ) {
				return;
			}

			add_filter( 'wp_list_table_class_name', array( $this, 'filter_list_table_class_name' ), 10, 2 );

			add_filter( 'handle_bulk_actions-users', array( $this, 'handle_site_bulk_actions' ), 10, 3 );
			add_filter( 'handle_network_bulk_actions-users-network', array( $this, 'handle_network_bulk_actions' ), 10, 3 );
			add_filter( 'handle_network_bulk_actions-site-users-network', array( $this, 'handle_site_users_network_bulk_actions' ), 10, 4 );

			add_action( 'admin_post_wpunconfirmed_action', array( $this, 'handle_row_action_request' ) );

			add_action( 'admin_notices', array( $this, 'maybe_render_admin_notice' ) );
			add_action( 'network_admin_notices', array( $this, 'maybe_render_admin_notice' ) );
		}

		/**
		 * @param string $class_name
		 * @param array  $args
		 * @return string
		 */
		public function filter_list_table_class_name( $class_name, $args ) {
			if ( 'WP_Users_List_Table' === $class_name && class_exists( 'WP_Unconfirmed_Users_List_Table' ) ) {
				return 'WP_Unconfirmed_Users_List_Table';
			}

			if ( 'WP_MS_Users_List_Table' === $class_name && class_exists( 'WP_Unconfirmed_MS_Users_List_Table' ) ) {
				return 'WP_Unconfirmed_MS_Users_List_Table';
			}

			return $class_name;
		}

		/**
		 * @return bool
		 */
		public function is_unconfirmed_request() {
			if ( ! isset( $_REQUEST['role'] ) ) {
				return false;
			}

			$role = sanitize_key( wp_unslash( $_REQUEST['role'] ) );

			return self::ROLE_FILTER === $role;
		}

		/**
		 * @param string $scope network|site
		 * @return bool
		 */
		public function current_user_can_manage_scope( $scope ) {
			if ( 'network' === $scope ) {
				return current_user_can( 'manage_network_users' );
			}

			if ( current_user_can( 'manage_sites' ) ) {
				return true;
			}

			return current_user_can( 'create_users' ) || current_user_can( 'promote_users' );
		}

		/**
		 * @param array<string,string> $views
		 * @param string               $base_url
		 * @param string               $scope network|site
		 * @param int                  $blog_id
		 * @return array<string,string>
		 */
		public function add_unconfirmed_view( array $views, $base_url, $scope, $blog_id ) {
			$count = $this->count_pending_signups( $scope, $blog_id );

			$url = add_query_arg(
				array(
					'role' => self::ROLE_FILTER,
				),
				$base_url
			);

			$label = sprintf(
				/* translators: %s: Number of unconfirmed invitations. */
				__( 'Unconfirmed <span class="count">(%s)</span>', 'wpunconfirmed' ),
				number_format_i18n( $count )
			);

			$current_attr = $this->is_unconfirmed_request() ? ' class="current" aria-current="page"' : '';
			$views[ self::ROLE_FILTER ] = sprintf(
				'<a href="%1$s"%2$s>%3$s</a>',
				esc_url( $url ),
				$current_attr,
				$label
			);

			return $views;
		}

		/**
		 * @return array<string,string>
		 */
		public function get_unconfirmed_bulk_actions() {
			return array(
				self::ACTION_RESEND   => __( 'Resend Invite', 'wpunconfirmed' ),
				self::ACTION_ACTIVATE => __( 'Activate Now', 'wpunconfirmed' ),
				self::ACTION_DELETE   => __( 'Delete Invitation', 'wpunconfirmed' ),
			);
		}

		/**
		 * @return array<string,string>
		 */
		public function get_unconfirmed_columns() {
			return array(
				'cb'       => '<input type="checkbox" />',
				'username' => __( 'Username' ),
				'email'    => __( 'Email' ),
				'site'     => __( 'Site' ),
				'invited'  => _x( 'Invited', 'user', 'wpunconfirmed' ),
				'status'   => __( 'Status', 'wpunconfirmed' ),
			);
		}

		/**
		 * @return array<string,array<int,mixed>>
		 */
		public function get_unconfirmed_sortable_columns() {
			return array(
				'username' => array( 'username', false ),
				'email'    => array( 'email', false ),
				'site'     => array( 'site', false ),
				'invited'  => array( 'invited', true ),
			);
		}

		/**
		 * @param string $scope network|site
		 * @param int    $blog_id
		 * @param string $search
		 * @param int    $paged
		 * @param int    $per_page
		 * @param string $orderby
		 * @param string $order
		 * @return array{items:array<int,stdClass>,total:int}
		 */
		public function get_pending_signups( $scope, $blog_id, $search, $paged, $per_page, $orderby, $order ) {
			$items = $this->load_pending_signups( $scope, $blog_id );

			if ( '' !== $search ) {
				$needle = wp_strtolower( $search );
				$items  = array_values(
					array_filter(
						$items,
						static function ( $item ) use ( $needle ) {
							$haystack = wp_strtolower( $item->user_login . ' ' . $item->user_email . ' ' . $item->site_label );
							return false !== strpos( $haystack, $needle );
						}
					)
				);
			}

			$orderby = $this->normalize_orderby( $orderby );
			$order   = $this->normalize_order( $order );

			usort(
				$items,
				static function ( $a, $b ) use ( $orderby, $order ) {
					$multiplier = ( 'asc' === $order ) ? 1 : -1;

					switch ( $orderby ) {
						case 'username':
							$cmp = strnatcasecmp( $a->user_login, $b->user_login );
							break;
						case 'email':
							$cmp = strnatcasecmp( $a->user_email, $b->user_email );
							break;
						case 'site':
							$cmp = strnatcasecmp( $a->site_label, $b->site_label );
							break;
						case 'invited':
						default:
							$cmp = strcmp( $a->registered, $b->registered );
							break;
					}

					return $cmp * $multiplier;
				}
			);

			$total  = count( $items );
			$offset = max( 0, ( $paged - 1 ) * $per_page );
			$items  = array_slice( $items, $offset, $per_page );

			return array(
				'items' => $items,
				'total' => $total,
			);
		}

		/**
		 * @param string $scope network|site
		 * @param int    $blog_id
		 * @return int
		 */
		public function count_pending_signups( $scope, $blog_id ) {
			return count( $this->load_pending_signups( $scope, $blog_id ) );
		}

		/**
		 * @param string $scope network|site
		 * @param int    $blog_id
		 * @return array<int,stdClass>
		 */
		private function load_pending_signups( $scope, $blog_id ) {
			$cache_key = $scope . ':' . (int) $blog_id;

			if ( isset( $this->pending_cache[ $cache_key ] ) ) {
				return $this->pending_cache[ $cache_key ];
			}

			if ( ! $this->can_read_signups_table() ) {
				$this->pending_cache[ $cache_key ] = array();
				return $this->pending_cache[ $cache_key ];
			}

			global $wpdb;

			$rows = $wpdb->get_results(
				"SELECT signup_id, user_login, user_email, registered, activation_key, meta, domain
				 FROM {$wpdb->signups}
				 WHERE active = 0
				 ORDER BY registered DESC"
			);

			$items = array();

			foreach ( $rows as $row ) {
				$meta = maybe_unserialize( $row->meta );
				if ( ! is_array( $meta ) ) {
					$meta = array();
				}

				$signup_blog_id = isset( $meta['add_to_blog'] ) ? absint( $meta['add_to_blog'] ) : 0;
				if ( 'site' === $scope && ( $signup_blog_id <= 0 || (int) $signup_blog_id !== (int) $blog_id ) ) {
					continue;
				}

				if ( ! empty( $row->domain ) && 0 === $signup_blog_id ) {
					continue;
				}

				$item                    = new stdClass();
				$item->signup_id         = (int) $row->signup_id;
				$item->user_login        = (string) $row->user_login;
				$item->user_email        = (string) $row->user_email;
				$item->registered        = (string) $row->registered;
				$item->activation_key    = (string) $row->activation_key;
				$item->signup_blog_id    = (int) $signup_blog_id;
				$item->signup_blog_label = $this->get_site_label( $signup_blog_id );
				$item->site_label        = $item->signup_blog_label;
				$item->status_label      = __( 'Unconfirmed', 'wpunconfirmed' );

				$items[] = $item;
			}

			$this->pending_cache[ $cache_key ] = $items;

			return $this->pending_cache[ $cache_key ];
		}

		/**
		 * @param int $blog_id
		 * @return string
		 */
		private function get_site_label( $blog_id ) {
			if ( $blog_id <= 0 ) {
				return __( 'Network registration', 'wpunconfirmed' );
			}

			if ( isset( $this->site_label_cache[ $blog_id ] ) ) {
				return $this->site_label_cache[ $blog_id ];
			}

			$details = get_blog_details( $blog_id );
			if ( ! $details ) {
				$this->site_label_cache[ $blog_id ] = sprintf(
					/* translators: %d: Site ID. */
					__( 'Site #%d', 'wpunconfirmed' ),
					$blog_id
				);
				return $this->site_label_cache[ $blog_id ];
			}

			$domain_path = $details->domain . untrailingslashit( $details->path );
			if ( '' === $domain_path ) {
				$domain_path = $details->domain;
			}

			$this->site_label_cache[ $blog_id ] = sprintf(
				/* translators: 1: Site title, 2: Site domain/path. */
				__( '%1$s (%2$s)', 'wpunconfirmed' ),
				$details->blogname,
				$domain_path
			);

			return $this->site_label_cache[ $blog_id ];
		}

		/**
		 * @return bool
		 */
		private function can_read_signups_table() {
			if ( ! is_multisite() ) {
				return false;
			}

			if ( null !== $this->signups_table_exists ) {
				return $this->signups_table_exists;
			}

			global $wpdb;
			if ( empty( $wpdb->signups ) ) {
				$this->signups_table_exists = false;
				return false;
			}

			$table = (string) $wpdb->signups;
			$found = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $table ) );

			$this->signups_table_exists = ( $found === $table );
			return $this->signups_table_exists;
		}

		/**
		 * @param string $orderby
		 * @return string
		 */
		private function normalize_orderby( $orderby ) {
			$orderby = sanitize_key( $orderby );
			$valid   = array( 'username', 'email', 'site', 'invited' );

			if ( in_array( $orderby, $valid, true ) ) {
				return $orderby;
			}

			return 'invited';
		}

		/**
		 * @param string $order
		 * @return string
		 */
		private function normalize_order( $order ) {
			$order = strtolower( $order );

			return ( 'asc' === $order ) ? 'asc' : 'desc';
		}

		/**
		 * @param stdClass $item
		 * @return string
		 */
		public function render_username_column( $item ) {
			$avatar = get_avatar( $item->user_email, 32 );

			return sprintf(
				'%1$s <strong>%2$s</strong>',
				$avatar,
				esc_html( $item->user_login )
			);
		}

		/**
		 * @param stdClass $item
		 * @return string
		 */
		public function render_email_column( $item ) {
			$email = esc_html( $item->user_email );
			return sprintf( '<a href="%1$s">%2$s</a>', esc_url( 'mailto:' . $item->user_email ), $email );
		}

		/**
		 * @param stdClass $item
		 * @return string
		 */
		public function render_site_column( $item ) {
			return esc_html( $item->site_label );
		}

		/**
		 * @param stdClass $item
		 * @return string
		 */
		public function render_invited_column( $item ) {
			return esc_html(
				mysql2date(
					get_option( 'date_format' ) . ' ' . get_option( 'time_format' ),
					$item->registered
				)
			);
		}

		/**
		 * @param stdClass $item
		 * @return string
		 */
		public function render_status_column( $item ) {
			return esc_html( $item->status_label );
		}

		/**
		 * @param string $operation resend|activate|delete
		 * @param int    $signup_id
		 * @param string $scope network|site
		 * @param int    $blog_id
		 * @return string
		 */
		public function get_row_action_url( $operation, $signup_id, $scope, $blog_id ) {
			$redirect_to = remove_query_arg( array( '_wpnonce' ), wp_unslash( $_SERVER['REQUEST_URI'] ) );

				$args = array(
					'action'      => 'wpunconfirmed_action',
					'operation'   => $operation,
					'signup_id'   => (int) $signup_id,
					'scope'       => ( 'network' === $scope ) ? 'network' : 'site',
					'blog_id'     => (int) $blog_id,
					'redirect_to' => $redirect_to,
				);

			$url = add_query_arg( $args, admin_url( 'admin-post.php' ) );

			return wp_nonce_url( $url, $this->row_action_nonce_action( $operation, $signup_id ) );
		}

		/**
		 * @param string $operation
		 * @param int    $signup_id
		 * @return string
		 */
		private function row_action_nonce_action( $operation, $signup_id ) {
			return 'wpunconfirmed_row_' . sanitize_key( $operation ) . '_' . (int) $signup_id;
		}

		/**
		 * @return void
		 */
		public function handle_row_action_request() {
			if ( ! current_user_can( 'list_users' ) ) {
				wp_die( esc_html__( 'Sorry, you are not allowed to list users.' ), 403 );
			}

			$operation = isset( $_REQUEST['operation'] ) ? sanitize_key( wp_unslash( $_REQUEST['operation'] ) ) : '';
			$signup_id = isset( $_REQUEST['signup_id'] ) ? absint( $_REQUEST['signup_id'] ) : 0;
			$scope     = ( isset( $_REQUEST['scope'] ) && 'network' === wp_unslash( $_REQUEST['scope'] ) ) ? 'network' : 'site';
			$blog_id   = isset( $_REQUEST['blog_id'] ) ? absint( $_REQUEST['blog_id'] ) : get_current_blog_id();

			if ( ! in_array( $operation, array( 'resend', 'activate', 'delete' ), true ) || $signup_id <= 0 ) {
				wp_die( esc_html__( 'Invalid invitation action.', 'wpunconfirmed' ), 400 );
			}

			check_admin_referer( $this->row_action_nonce_action( $operation, $signup_id ) );

			if ( ! $this->current_user_can_manage_scope( $scope ) ) {
				wp_die( esc_html__( 'Sorry, you are not allowed to manage unconfirmed users.', 'wpunconfirmed' ), 403 );
			}

			$result = $this->process_signup_action( $operation, array( $signup_id ), $scope, $blog_id );

				$redirect_to = isset( $_REQUEST['redirect_to'] ) ? wp_unslash( $_REQUEST['redirect_to'] ) : '';
			$fallback    = ( 'network' === $scope ) ? network_admin_url( 'users.php' ) : admin_url( 'users.php' );

			$redirect_to = wp_validate_redirect( $redirect_to, $fallback );
			$redirect_to = $this->add_result_query_args( $redirect_to, $operation, $result['done'], $result['failed'] );

			wp_safe_redirect( $redirect_to );
			exit;
		}

		/**
		 * @param string $sendback
		 * @param string $action
		 * @param int[]  $signup_ids
		 * @return string
		 */
		public function handle_site_bulk_actions( $sendback, $action, $signup_ids ) {
			$operation = $this->operation_from_bulk_action( $action );
			if ( '' === $operation ) {
				return $sendback;
			}

			if ( ! $this->current_user_can_manage_scope( 'site' ) ) {
				return $sendback;
			}

			if ( ! isset( $_REQUEST['_wpnonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_REQUEST['_wpnonce'] ) ), 'bulk-users' ) ) {
				return $this->add_result_query_args( $sendback, $operation, 0, count( (array) $signup_ids ) );
			}

			$result = $this->process_signup_action( $operation, (array) $signup_ids, 'site', get_current_blog_id() );

			return $this->add_result_query_args( $sendback, $operation, $result['done'], $result['failed'] );
		}

		/**
		 * @param string $sendback
		 * @param string $action
		 * @param int[]  $signup_ids
		 * @return string
		 */
		public function handle_network_bulk_actions( $sendback, $action, $signup_ids ) {
			$operation = $this->operation_from_bulk_action( $action );
			if ( '' === $operation ) {
				return $sendback;
			}

			if ( ! $this->current_user_can_manage_scope( 'network' ) ) {
				return $sendback;
			}

			$result = $this->process_signup_action( $operation, (array) $signup_ids, 'network', 0 );

			return $this->add_result_query_args( $sendback, $operation, $result['done'], $result['failed'] );
		}

		/**
		 * @param string $sendback
		 * @param string $action
		 * @param int[]  $signup_ids
		 * @param int    $site_id
		 * @return string
		 */
		public function handle_site_users_network_bulk_actions( $sendback, $action, $signup_ids, $site_id ) {
			$operation = $this->operation_from_bulk_action( $action );
			if ( '' === $operation ) {
				return $sendback;
			}

			if ( ! $this->current_user_can_manage_scope( 'site' ) ) {
				return $sendback;
			}

			$result = $this->process_signup_action( $operation, (array) $signup_ids, 'site', (int) $site_id );

			return $this->add_result_query_args( $sendback, $operation, $result['done'], $result['failed'] );
		}

		/**
		 * @param string $action
		 * @return string
		 */
		private function operation_from_bulk_action( $action ) {
			switch ( $action ) {
				case self::ACTION_RESEND:
					return 'resend';
				case self::ACTION_ACTIVATE:
					return 'activate';
				case self::ACTION_DELETE:
					return 'delete';
				default:
					return '';
			}
		}

		/**
		 * @param string $operation resend|activate|delete
		 * @param int[]  $signup_ids
		 * @param string $scope network|site
		 * @param int    $blog_id
		 * @return array{done:int,failed:int}
		 */
		private function process_signup_action( $operation, array $signup_ids, $scope, $blog_id ) {
			$done   = 0;
			$failed = 0;

			foreach ( $signup_ids as $signup_id ) {
				$signup_id = absint( $signup_id );
				if ( $signup_id <= 0 ) {
					continue;
				}

				$result = $this->process_single_signup( $operation, $signup_id, $scope, $blog_id );
				if ( true === $result ) {
					++$done;
				} else {
					++$failed;
				}
			}

			return array(
				'done'   => $done,
				'failed' => $failed,
			);
		}

		/**
		 * @param string $operation resend|activate|delete
		 * @param int    $signup_id
		 * @param string $scope network|site
		 * @param int    $blog_id
		 * @return bool
		 */
		private function process_single_signup( $operation, $signup_id, $scope, $blog_id ) {
			$signup = $this->get_pending_signup_by_id( $signup_id );
			if ( ! $signup ) {
				return false;
			}

			if ( 'site' === $scope && ! $this->signup_matches_blog( $signup, $blog_id ) ) {
				return false;
			}

			switch ( $operation ) {
				case 'resend':
					return $this->resend_signup_email( $signup );
				case 'activate':
					return $this->activate_signup( $signup );
				case 'delete':
					return $this->delete_signup( $signup );
				default:
					return false;
			}
		}

		/**
		 * @param stdClass $signup
		 * @param int      $blog_id
		 * @return bool
		 */
		private function signup_matches_blog( $signup, $blog_id ) {
			$meta = maybe_unserialize( $signup->meta );
			if ( ! is_array( $meta ) || empty( $meta['add_to_blog'] ) ) {
				return false;
			}

			return (int) absint( $meta['add_to_blog'] ) === (int) absint( $blog_id );
		}

		/**
		 * @param int $signup_id
		 * @return stdClass|null
		 */
		private function get_pending_signup_by_id( $signup_id ) {
			if ( ! $this->can_read_signups_table() ) {
				return null;
			}

			global $wpdb;
			$signup = $wpdb->get_row(
				$wpdb->prepare(
					"SELECT signup_id, user_login, user_email, registered, activation_key, meta, active
					 FROM {$wpdb->signups}
					 WHERE signup_id = %d",
					$signup_id
				)
			);

			if ( ! $signup || 1 === (int) $signup->active ) {
				return null;
			}

			return $signup;
		}

		/**
		 * @param stdClass $signup
		 * @return bool
		 */
		private function resend_signup_email( $signup ) {
			if ( ! is_multisite() || ! function_exists( 'wpmu_signup_user_notification' ) ) {
				return false;
			}

			$meta = maybe_unserialize( $signup->meta );
			if ( ! is_array( $meta ) ) {
				$meta = array();
			}

			wpmu_signup_user_notification( $signup->user_login, $signup->user_email, $signup->activation_key, $meta );

			return true;
		}

		/**
		 * @param stdClass $signup
		 * @return bool
		 */
		private function activate_signup( $signup ) {
			if ( ! is_multisite() || ! function_exists( 'wpmu_activate_signup' ) ) {
				return false;
			}

			$result = wpmu_activate_signup( $signup->activation_key );
			if ( is_wp_error( $result ) ) {
				$code = $result->get_error_code();

				if ( 'already_active' === $code ) {
					return true;
				}

				if ( 'user_already_exists' === $code ) {
					return $this->activate_existing_user_signup( $signup );
				}

				return false;
			}

			return true;
		}

		/**
		 * Handles pending signups where the user already exists but signup remains unconfirmed.
		 *
		 * @param stdClass $signup
		 * @return bool
		 */
		private function activate_existing_user_signup( $signup ) {
			global $wpdb;

			$user = get_user_by( 'login', $signup->user_login );
			if ( ! $user ) {
				$user = get_user_by( 'email', $signup->user_email );
			}
			if ( ! $user ) {
				return false;
			}

			$meta = maybe_unserialize( $signup->meta );
			if ( ! is_array( $meta ) ) {
				$meta = array();
			}

			if ( ! empty( $meta['add_to_blog'] ) ) {
				$blog_id = absint( $meta['add_to_blog'] );
				$role    = ! empty( $meta['new_role'] ) ? sanitize_key( $meta['new_role'] ) : get_option( 'default_role', 'subscriber' );

				if ( $blog_id > 0 && ! is_user_member_of_blog( $user->ID, $blog_id ) ) {
					$result = add_user_to_blog( $blog_id, $user->ID, $role );
					if ( is_wp_error( $result ) ) {
						return false;
					}
				}
			}

			$updated = $wpdb->update(
				$wpdb->signups,
				array(
					'active'    => 1,
					'activated' => current_time( 'mysql', true ),
				),
				array( 'signup_id' => (int) $signup->signup_id ),
				array( '%d', '%s' ),
				array( '%d' )
			);

			return false !== $updated;
		}

		/**
		 * @param stdClass $signup
		 * @return bool
		 */
		private function delete_signup( $signup ) {
			global $wpdb;

			$deleted = $wpdb->delete(
				$wpdb->signups,
				array( 'signup_id' => (int) $signup->signup_id ),
				array( '%d' )
			);

			return false !== $deleted;
		}

		/**
		 * @param string $sendback
		 * @param string $operation
		 * @param int    $done
		 * @param int    $failed
		 * @return string
		 */
		private function add_result_query_args( $sendback, $operation, $done, $failed ) {
			$sendback = remove_query_arg(
				array(
					'wpunconfirmed_op',
					'wpunconfirmed_done',
					'wpunconfirmed_failed',
				),
				$sendback
			);

			return add_query_arg(
				array(
					'wpunconfirmed_op'     => sanitize_key( $operation ),
					'wpunconfirmed_done'   => (int) $done,
					'wpunconfirmed_failed' => (int) $failed,
				),
				$sendback
			);
		}

		/**
		 * @return void
		 */
		public function maybe_render_admin_notice() {
			$screen = get_current_screen();
			if ( ! $screen || ! in_array( $screen->id, array( 'users', 'users-network', 'site-users-network' ), true ) ) {
				return;
			}

			if ( ! isset( $_GET['wpunconfirmed_op'], $_GET['wpunconfirmed_done'], $_GET['wpunconfirmed_failed'] ) ) {
				return;
			}

			$operation = sanitize_key( wp_unslash( $_GET['wpunconfirmed_op'] ) );
			$done      = max( 0, (int) $_GET['wpunconfirmed_done'] );
			$failed    = max( 0, (int) $_GET['wpunconfirmed_failed'] );
			$total     = $done + $failed;

			if ( $total <= 0 ) {
				return;
			}

			$label_map = array(
				'resend'   => __( 'Invitation emails resent.', 'wpunconfirmed' ),
				'activate' => __( 'Invitations activated.', 'wpunconfirmed' ),
				'delete'   => __( 'Invitations deleted.', 'wpunconfirmed' ),
			);

			$title = isset( $label_map[ $operation ] ) ? $label_map[ $operation ] : __( 'Unconfirmed invitations updated.', 'wpunconfirmed' );

			if ( $failed > 0 ) {
				$message = sprintf(
					/* translators: 1: Success count, 2: Failure count. */
					__( '%1$d succeeded, %2$d failed.', 'wpunconfirmed' ),
					$done,
					$failed
				);
				$class = 'notice notice-warning is-dismissible';
			} else {
				$message = sprintf(
					/* translators: %d: Success count. */
					_n( '%d invitation updated.', '%d invitations updated.', $done, 'wpunconfirmed' ),
					$done
				);
				$class = 'notice notice-success is-dismissible';
			}
			?>
			<div class="<?php echo esc_attr( $class ); ?>">
				<p><strong><?php echo esc_html( $title ); ?></strong> <?php echo esc_html( $message ); ?></p>
			</div>
			<?php
		}
	}
}

if ( is_admin() ) {
	require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
	require_once ABSPATH . 'wp-admin/includes/class-wp-users-list-table.php';
	require_once ABSPATH . 'wp-admin/includes/class-wp-ms-users-list-table.php';
}

if ( is_admin() && ! class_exists( 'WP_Unconfirmed_Users_List_Table' ) ) {
	class WP_Unconfirmed_Users_List_Table extends WP_Users_List_Table {
		/** @var bool */
		private $unconfirmed_mode = false;

		/** @var int */
		private $context_blog_id = 0;

		public function __construct( $args = array() ) {
			parent::__construct( $args );

			$this->unconfirmed_mode = WP_Unconfirmed_Users_Plugin::instance()->is_unconfirmed_request();
			$this->context_blog_id  = $this->is_site_users ? (int) $this->site_id : get_current_blog_id();
		}

		public function prepare_items() {
			if ( ! $this->unconfirmed_mode ) {
				parent::prepare_items();
				return;
			}

			global $role, $usersearch;

			$role       = isset( $_REQUEST['role'] ) ? sanitize_key( wp_unslash( $_REQUEST['role'] ) ) : '';
			$usersearch = isset( $_REQUEST['s'] ) ? wp_unslash( trim( $_REQUEST['s'] ) ) : '';

			$per_page_key = $this->is_site_users ? 'site_users_network_per_page' : 'users_per_page';
			$per_page     = $this->get_items_per_page( $per_page_key );
			$paged        = $this->get_pagenum();

			$orderby = isset( $_REQUEST['orderby'] ) ? wp_unslash( $_REQUEST['orderby'] ) : 'invited';
			$order   = isset( $_REQUEST['order'] ) ? wp_unslash( $_REQUEST['order'] ) : 'desc';

			$data = WP_Unconfirmed_Users_Plugin::instance()->get_pending_signups(
				'site',
				$this->context_blog_id,
				(string) $usersearch,
				(int) $paged,
				(int) $per_page,
				(string) $orderby,
				(string) $order
			);

			$this->items = $data['items'];

			$this->set_pagination_args(
				array(
					'total_items' => $data['total'],
					'per_page'    => $per_page,
				)
			);
		}

		protected function get_views() {
			$views = parent::get_views();

			$base_url = $this->is_site_users
				? add_query_arg( 'id', (int) $this->site_id, 'site-users.php' )
				: 'users.php';

			return WP_Unconfirmed_Users_Plugin::instance()->add_unconfirmed_view( $views, $base_url, 'site', $this->context_blog_id );
		}

		protected function get_bulk_actions() {
			if ( ! $this->unconfirmed_mode ) {
				return parent::get_bulk_actions();
			}

			return WP_Unconfirmed_Users_Plugin::instance()->get_unconfirmed_bulk_actions();
		}

		public function get_columns() {
			if ( ! $this->unconfirmed_mode ) {
				return parent::get_columns();
			}

			return WP_Unconfirmed_Users_Plugin::instance()->get_unconfirmed_columns();
		}

		protected function get_sortable_columns() {
			if ( ! $this->unconfirmed_mode ) {
				return parent::get_sortable_columns();
			}

			return WP_Unconfirmed_Users_Plugin::instance()->get_unconfirmed_sortable_columns();
		}

		public function no_items() {
			if ( ! $this->unconfirmed_mode ) {
				parent::no_items();
				return;
			}

			if ( ! is_multisite() ) {
				echo esc_html__( 'No unconfirmed users found. WordPress core stores pending invitations in multisite installs.', 'wpunconfirmed' );
				return;
			}

			echo esc_html__( 'No unconfirmed users found.', 'wpunconfirmed' );
		}

		public function display_rows() {
			if ( ! $this->unconfirmed_mode ) {
				parent::display_rows();
				return;
			}

			foreach ( $this->items as $item ) {
				echo '<tr>';
				$this->single_row_columns( $item );
				echo '</tr>';
			}
		}

		public function column_cb( $item ) {
			if ( ! WP_Unconfirmed_Users_Plugin::instance()->current_user_can_manage_scope( 'site' ) ) {
				return '';
			}

			return sprintf(
				'<input type="checkbox" name="users[]" id="signup_%1$d" value="%1$d" />' .
				'<label for="signup_%1$d"><span class="screen-reader-text">%2$s</span></label>',
				(int) $item->signup_id,
					esc_html( sprintf( __( 'Select %s', 'wpunconfirmed' ), $item->user_login ) )
			);
		}

		public function column_username( $item ) {
			return WP_Unconfirmed_Users_Plugin::instance()->render_username_column( $item );
		}

		public function column_email( $item ) {
			return WP_Unconfirmed_Users_Plugin::instance()->render_email_column( $item );
		}

		public function column_site( $item ) {
			return WP_Unconfirmed_Users_Plugin::instance()->render_site_column( $item );
		}

		public function column_invited( $item ) {
			return WP_Unconfirmed_Users_Plugin::instance()->render_invited_column( $item );
		}

		public function column_status( $item ) {
			return WP_Unconfirmed_Users_Plugin::instance()->render_status_column( $item );
		}

		protected function handle_row_actions( $item, $column_name, $primary ) {
			if ( ! $this->unconfirmed_mode ) {
				return parent::handle_row_actions( $item, $column_name, $primary );
			}

			if ( $primary !== $column_name ) {
				return '';
			}

			$plugin = WP_Unconfirmed_Users_Plugin::instance();
			if ( ! $plugin->current_user_can_manage_scope( 'site' ) ) {
				return '';
			}

			$actions = array(
				'resend'   => sprintf(
					'<a href="%s">%s</a>',
					esc_url( $plugin->get_row_action_url( 'resend', (int) $item->signup_id, 'site', $this->context_blog_id ) ),
					esc_html__( 'Resend Invite', 'wpunconfirmed' )
				),
				'activate' => sprintf(
					'<a href="%s">%s</a>',
					esc_url( $plugin->get_row_action_url( 'activate', (int) $item->signup_id, 'site', $this->context_blog_id ) ),
					esc_html__( 'Activate', 'wpunconfirmed' )
				),
				'delete'   => sprintf(
					'<a class="delete" href="%s" onclick="return confirm(%s);">%s</a>',
					esc_url( $plugin->get_row_action_url( 'delete', (int) $item->signup_id, 'site', $this->context_blog_id ) ),
					wp_json_encode( __( 'Delete this invitation?', 'wpunconfirmed' ) ),
					esc_html__( 'Delete', 'wpunconfirmed' )
				),
			);

			return $this->row_actions( $actions );
		}

		protected function get_default_primary_column_name() {
			if ( ! $this->unconfirmed_mode ) {
				return parent::get_default_primary_column_name();
			}

			return 'username';
		}
	}
}

if ( is_admin() && ! class_exists( 'WP_Unconfirmed_MS_Users_List_Table' ) ) {
	class WP_Unconfirmed_MS_Users_List_Table extends WP_MS_Users_List_Table {
		/** @var bool */
		private $unconfirmed_mode = false;

		public function __construct( $args = array() ) {
			parent::__construct( $args );
			$this->unconfirmed_mode = WP_Unconfirmed_Users_Plugin::instance()->is_unconfirmed_request();
		}

		public function prepare_items() {
			if ( ! $this->unconfirmed_mode ) {
				parent::prepare_items();
				return;
			}

			global $mode, $usersearch, $role;

			if ( ! empty( $_REQUEST['mode'] ) ) {
				$mode = ( 'excerpt' === $_REQUEST['mode'] ) ? 'excerpt' : 'list';
				set_user_setting( 'network_users_list_mode', $mode );
			} else {
				$mode = get_user_setting( 'network_users_list_mode', 'list' );
			}

			$usersearch = isset( $_REQUEST['s'] ) ? wp_unslash( trim( $_REQUEST['s'] ) ) : '';
			$role       = isset( $_REQUEST['role'] ) ? sanitize_key( wp_unslash( $_REQUEST['role'] ) ) : '';

			$per_page = $this->get_items_per_page( 'users_network_per_page' );
			$paged    = $this->get_pagenum();

			$orderby = isset( $_REQUEST['orderby'] ) ? wp_unslash( $_REQUEST['orderby'] ) : 'invited';
			$order   = isset( $_REQUEST['order'] ) ? wp_unslash( $_REQUEST['order'] ) : 'desc';

			$data = WP_Unconfirmed_Users_Plugin::instance()->get_pending_signups(
				'network',
				0,
				(string) $usersearch,
				(int) $paged,
				(int) $per_page,
				(string) $orderby,
				(string) $order
			);

			$this->items = $data['items'];

			$this->set_pagination_args(
				array(
					'total_items' => $data['total'],
					'per_page'    => $per_page,
				)
			);
		}

		protected function get_views() {
			$views = parent::get_views();
			return WP_Unconfirmed_Users_Plugin::instance()->add_unconfirmed_view( $views, network_admin_url( 'users.php' ), 'network', 0 );
		}

		protected function get_bulk_actions() {
			if ( ! $this->unconfirmed_mode ) {
				return parent::get_bulk_actions();
			}

			return WP_Unconfirmed_Users_Plugin::instance()->get_unconfirmed_bulk_actions();
		}

		public function get_columns() {
			if ( ! $this->unconfirmed_mode ) {
				return parent::get_columns();
			}

			return WP_Unconfirmed_Users_Plugin::instance()->get_unconfirmed_columns();
		}

		protected function get_sortable_columns() {
			if ( ! $this->unconfirmed_mode ) {
				return parent::get_sortable_columns();
			}

			return WP_Unconfirmed_Users_Plugin::instance()->get_unconfirmed_sortable_columns();
		}

		public function no_items() {
			if ( ! $this->unconfirmed_mode ) {
				parent::no_items();
				return;
			}

			echo esc_html__( 'No unconfirmed users found.', 'wpunconfirmed' );
		}

		public function display_rows() {
			if ( ! $this->unconfirmed_mode ) {
				parent::display_rows();
				return;
			}

			foreach ( $this->items as $item ) {
				echo '<tr>';
				$this->single_row_columns( $item );
				echo '</tr>';
			}
		}

		public function column_cb( $item ) {
			if ( ! WP_Unconfirmed_Users_Plugin::instance()->current_user_can_manage_scope( 'network' ) ) {
				return '';
			}
			?>
			<input type="checkbox" id="signup_<?php echo (int) $item->signup_id; ?>" name="allusers[]" value="<?php echo esc_attr( (int) $item->signup_id ); ?>" />
			<label for="signup_<?php echo (int) $item->signup_id; ?>">
				<span class="screen-reader-text">
					<?php
					printf(
						/* translators: %s: Username. */
							esc_html__( 'Select %s', 'wpunconfirmed' ),
						esc_html( $item->user_login )
					);
					?>
				</span>
			</label>
			<?php
		}

		public function column_username( $item ) {
			return WP_Unconfirmed_Users_Plugin::instance()->render_username_column( $item );
		}

		public function column_email( $item ) {
			return WP_Unconfirmed_Users_Plugin::instance()->render_email_column( $item );
		}

		public function column_site( $item ) {
			return WP_Unconfirmed_Users_Plugin::instance()->render_site_column( $item );
		}

		public function column_invited( $item ) {
			return WP_Unconfirmed_Users_Plugin::instance()->render_invited_column( $item );
		}

		public function column_status( $item ) {
			return WP_Unconfirmed_Users_Plugin::instance()->render_status_column( $item );
		}

		protected function handle_row_actions( $item, $column_name, $primary ) {
			if ( ! $this->unconfirmed_mode ) {
				return parent::handle_row_actions( $item, $column_name, $primary );
			}

			if ( $primary !== $column_name ) {
				return '';
			}

			$plugin = WP_Unconfirmed_Users_Plugin::instance();
			if ( ! $plugin->current_user_can_manage_scope( 'network' ) ) {
				return '';
			}

			$actions = array(
				'resend'   => sprintf(
					'<a href="%s">%s</a>',
					esc_url( $plugin->get_row_action_url( 'resend', (int) $item->signup_id, 'network', 0 ) ),
					esc_html__( 'Resend Invite', 'wpunconfirmed' )
				),
				'activate' => sprintf(
					'<a href="%s">%s</a>',
					esc_url( $plugin->get_row_action_url( 'activate', (int) $item->signup_id, 'network', 0 ) ),
					esc_html__( 'Activate', 'wpunconfirmed' )
				),
				'delete'   => sprintf(
					'<a class="delete" href="%s" onclick="return confirm(%s);">%s</a>',
					esc_url( $plugin->get_row_action_url( 'delete', (int) $item->signup_id, 'network', 0 ) ),
					wp_json_encode( __( 'Delete this invitation?', 'wpunconfirmed' ) ),
					esc_html__( 'Delete', 'wpunconfirmed' )
				),
			);

			return $this->row_actions( $actions );
		}

		protected function get_default_primary_column_name() {
			if ( ! $this->unconfirmed_mode ) {
				return parent::get_default_primary_column_name();
			}

			return 'username';
		}
	}
}

WP_Unconfirmed_Users_Plugin::instance();
